"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareSecuritySchemes = exports.getAllAuthorizers = exports.serializeAsAuthorizerReference = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const utils_1 = require("./utils");
const authorizers_1 = require("../authorizers");
const predicates_1 = require("../authorizers/predicates");
// Regex to match against a single header identity source
const SINGLE_HEADER_IDENTITY_SOURCE_REGEX = /^method.request.header.(?<header>[^\.\s,]+)$/;
/**
 * Build a serialized reference to an authorizer for use in an api method
 * @param authorizer the author to serialize
 */
const serializeAsAuthorizerReference = (authorizer) => authorizer
    ? {
        authorizerId: authorizer.authorizerId,
        authorizationScopes: authorizer.authorizationScopes,
    }
    : undefined;
exports.serializeAsAuthorizerReference = serializeAsAuthorizerReference;
/**
 * Create an OpenAPI security scheme definition for an iam authorizer
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/api-gateway-swagger-extensions-authtype.html
 */
const iamSecurityScheme = () => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": "awsSigv4",
});
/**
 * Create an OpenAPI security scheme definition for a cognito authorizer
 * @param authorizer cognito authorizer
 */
const cognitoSecurityScheme = (authorizer) => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": authorizer.authorizationType,
    "x-amazon-apigateway-authorizer": {
        type: aws_apigateway_1.AuthorizationType.COGNITO,
        providerARNs: authorizer.userPools.map((pool) => pool.userPoolArn),
    },
});
/**
 * Create an OpenAPI security scheme definition for a custom authorizer
 * @param scope the scope in which the api construct is defined
 * @param authorizer custom authorizer
 */
const customSecurityScheme = (scope, authorizer) => {
    const singleHeaderMatch = authorizer.identitySource.match(SINGLE_HEADER_IDENTITY_SOURCE_REGEX);
    const singleHeader = singleHeaderMatch
        ? singleHeaderMatch.groups.header
        : undefined;
    // Token type must use a single header
    if (!singleHeader && authorizer.type === authorizers_1.CustomAuthorizerType.TOKEN) {
        throw new Error(`identitySource must be a single header for a ${authorizers_1.CustomAuthorizerType.TOKEN} authorizer, eg method.request.header.Authorization`);
    }
    return {
        type: "apiKey",
        in: "header",
        // When the identity source is not a single header, the value must be "Unused"
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-authorizer.html
        name: singleHeader ? singleHeader : "Unused",
        "x-amazon-apigateway-authtype": authorizer.authorizationType,
        "x-amazon-apigateway-authorizer": {
            type: authorizer.type,
            authorizerUri: (0, utils_1.functionInvocationUri)(scope, authorizer.function),
            authorizerResultTtlInSeconds: authorizer.authorizerResultTtlInSeconds,
            identitySource: authorizer.identitySource,
        },
    };
};
/**
 * Return a list of all unique authorizers used in the api
 */
const getAllAuthorizers = (integrations, defaultAuthorizer) => Object.values(Object.fromEntries([
    ...(defaultAuthorizer ? [defaultAuthorizer] : []),
    ...Object.values(integrations).flatMap(({ authorizer }) => authorizer ? [authorizer] : []),
].map((a) => [a.authorizerId, a])));
exports.getAllAuthorizers = getAllAuthorizers;
/**
 * Generate the security schemes section of an OpenAPI specification
 */
const prepareSecuritySchemes = (scope, integrations, defaultAuthorizer) => {
    // All the defined authorizers
    const allAuthorizers = (0, exports.getAllAuthorizers)(integrations, defaultAuthorizer);
    // Cognito, IAM and custom authorizers must be declared in security schemes
    return {
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isCognitoAuthorizer)(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            cognitoSecurityScheme(authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isCustomAuthorizer)(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            customSecurityScheme(scope, authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isIamAuthorizer)(authorizer))
            .map((authorizer) => [authorizer.authorizerId, iamSecurityScheme()])),
    };
};
exports.prepareSecuritySchemes = prepareSecuritySchemes;
//# sourceMappingURL=data:application/json;base64,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