"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
const concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
exports.concatMethodAndPath = concatMethodAndPath;
// Add to methods to ensure no auth is added
const NO_AUTH_SPEC_SNIPPET = {
    security: [],
    "x-amazon-apigateway-auth": {
        type: "NONE",
    },
};
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 */
const applyMethodAuthorizer = (methodAuthorizer) => {
    if (methodAuthorizer) {
        if (methodAuthorizer.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return NO_AUTH_SPEC_SNIPPET;
        }
        else {
            return {
                security: [
                    {
                        [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                    },
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { methodAuthorizer, integration } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": integration,
        ...applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
            // No auth for CORS options requests
            ...NO_AUTH_SPEC_SNIPPET,
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    const supportedHttpMethods = new Set(Object.values(constants_1.HttpMethods));
    const unsupportedMethodsInSpec = Object.keys(pathItem).filter((method) => !supportedHttpMethods.has(method));
    if (unsupportedMethodsInSpec.length > 0) {
        throw new Error(`Path ${path} contains unsupported method${unsupportedMethodsInSpec.length > 1 ? "s" : ""} ${unsupportedMethodsInSpec.join(", ")}. Supported methods are ${Object.values(constants_1.HttpMethods).join(", ")}.`);
    }
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
const prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        (0, exports.concatMethodAndPath)(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[(0, exports.concatMethodAndPath)(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...options.securitySchemes,
            },
        },
        // Apply the default authorizer at the top level
        ...(options.defaultAuthorizerReference
            ? applyMethodAuthorizer(options.defaultAuthorizerReference)
            : {}),
    };
};
exports.prepareApiSpec = prepareApiSpec;
//# sourceMappingURL=data:application/json;base64,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