"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": [
        "export * from './api';",
        ...(options.sampleCode !== false ? ["export * from './sample-api';"] : []),
    ].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayRestApi construct
    "api.ts": `import { OpenApiGatewayRestApi, OpenApiGatewayRestApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import * as path from "path";
import spec from "../${options.specDir}/${options.parsedSpecFileName}";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayRestApiProps, "spec" | "specPath" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
export class Api extends OpenApiGatewayRestApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      specPath: path.resolve(__dirname, "../${options.specDir}/${options.parsedSpecFileName}"),
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Authorizers, Integrations } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { Cors } from "aws-cdk-lib/aws-apigateway";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      defaultAuthorizer: Authorizers.iam(),
      corsOptions: {
        allowOrigins: Cors.ALL_ORIGINS,
        allowMethods: Cors.ALL_METHODS,
      },
      integrations: {
        sayHello: {
          integration: Integrations.lambda(new NodejsFunction(scope, "say-hello")),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "sample-api.say-hello.ts": `import { sayHelloHandler } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler(async ({ input }) => {
  return {
    statusCode: 200,
    headers: {
      "Access-Control-Allow-Origin": "*",
      "Access-Control-Allow-Headers": "*",
    },
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Integrations } from "${options.openApiGatewayPackageName}";
import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.apiSrcDir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        integration: Integrations.lambda(new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        })),
      }])) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,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