"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d;
        this.props = props;
        Bundling.runsLocally = (_b = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : util_1.getGoBuildVersion()) !== null && _b !== void 0 ? _b : false;
        const projectRoot = path.dirname(props.moduleDir);
        this.relativeEntryPath = `./${path.relative(projectRoot, path.resolve(props.entry))}`;
        const cgoEnabled = props.cgoEnabled ? '1' : '0';
        const environment = {
            CGO_ENABLED: cgoEnabled,
            GO111MODULE: 'on',
            GOARCH: props.architecture.dockerPlatform.split('/')[1],
            GOOS: 'linux',
            ...props.environment,
        };
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_c = props.dockerImage) !== null && _c !== void 0 ? _c : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_d = props.buildArgs) !== null && _d !== void 0 ? _d : {},
                    IMAGE: aws_lambda_1.Runtime.GO_1_X.bundlingImage.image,
                },
                platform: props.architecture.dockerPlatform,
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = environment;
        // Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('go build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...environment !== null && environment !== void 0 ? environment : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.moduleDir),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    static bundle(options) {
        var _a;
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(path.dirname(options.moduleDir), {
            assetHashType: (_a = options.assetHashType) !== null && _a !== void 0 ? _a : cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d;
        const pathJoin = osPathJoin(osPlatform);
        const hasVendor = util_1.findUp('vendor', path.dirname(this.props.entry));
        const goBuildCommand = [
            'go', 'build',
            hasVendor ? '-mod=vendor' : '',
            '-o', `${pathJoin(outputDir, 'bootstrap')}`,
            `${this.props.goBuildFlags ? this.props.goBuildFlags.join(' ') : ''}`,
            `${this.relativeEntryPath.replace(/\\/g, '/')}`,
        ].filter(c => !!c).join(' ');
        return chain([
            ...(_b = (_a = this.props.commandHooks) === null || _a === void 0 ? void 0 : _a.beforeBundling(inputDir, outputDir)) !== null && _b !== void 0 ? _b : [],
            goBuildCommand,
            ...(_d = (_c = this.props.commandHooks) === null || _c === void 0 ? void 0 : _c.afterBundling(inputDir, outputDir)) !== null && _d !== void 0 ? _d : [],
        ]);
    }
}
exports.Bundling = Bundling;
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG9EQUE2RTtBQUM3RSxxQ0FBcUM7QUFFckMsaUNBQXlEO0FBMkR6RDs7R0FFRztBQUNILE1BQWEsUUFBUTtJQStCbkIsWUFBNkIsS0FBb0I7O1FBQXBCLFVBQUssR0FBTCxLQUFLLENBQWU7UUFDL0MsUUFBUSxDQUFDLFdBQVcsZUFBRyxRQUFRLENBQUMsV0FBVyxtQ0FDdEMsd0JBQWlCLEVBQUUsbUNBQ25CLEtBQUssQ0FBQztRQUVYLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2xELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUV0RixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQztRQUVoRCxNQUFNLFdBQVcsR0FBRztZQUNsQixXQUFXLEVBQUUsVUFBVTtZQUN2QixXQUFXLEVBQUUsSUFBSTtZQUNqQixNQUFNLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN2RCxJQUFJLEVBQUUsT0FBTztZQUNiLEdBQUcsS0FBSyxDQUFDLFdBQVc7U0FDckIsQ0FBQztRQUVGLGtCQUFrQjtRQUNsQixNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUM7UUFDN0UsSUFBSSxDQUFDLEtBQUssR0FBRyxnQkFBZ0I7WUFDM0IsQ0FBQyxPQUFDLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxFQUFFO2dCQUMvRSxTQUFTLEVBQUU7b0JBQ1QsU0FBRyxLQUFLLENBQUMsU0FBUyxtQ0FBSSxFQUFFO29CQUN4QixLQUFLLEVBQUUsb0JBQU8sQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEtBQUs7aUJBQzFDO2dCQUNELFFBQVEsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLGNBQWM7YUFDNUMsQ0FBQyxDQUNGLENBQUMsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLG1DQUFtQztRQUU5RSxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUgsSUFBSSxDQUFDLE9BQU8sR0FBRyxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUM7UUFFL0IsaUJBQWlCO1FBQ2pCLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLEVBQUUsRUFBRSwrQkFBK0I7WUFFaEUsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ2pDLE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxTQUFpQixFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsV0FBVyxFQUFFLFNBQVMsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUVqSCxJQUFJLENBQUMsS0FBSyxHQUFHO2dCQUNYLFNBQVMsQ0FBQyxTQUFpQjtvQkFDekIsSUFBSSxRQUFRLENBQUMsV0FBVyxJQUFJLEtBQUssRUFBRTt3QkFDakMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsOERBQThELENBQUMsQ0FBQzt3QkFDckYsT0FBTyxLQUFLLENBQUM7cUJBQ2Q7b0JBRUQsTUFBTSxZQUFZLEdBQUcsa0JBQWtCLENBQUMsU0FBUyxDQUFDLENBQUM7b0JBQ25ELFdBQUksQ0FDRixVQUFVLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sRUFDdkM7d0JBQ0UsVUFBVSxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJO3dCQUNwQyxZQUFZO3FCQUNiLEVBQ0Q7d0JBQ0UsR0FBRyxFQUFFLEVBQUUsR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsV0FBVyxhQUFYLFdBQVcsY0FBWCxXQUFXLEdBQUksRUFBRSxFQUFFO3dCQUM3QyxLQUFLLEVBQUU7NEJBQ0wsUUFBUTs0QkFDUixPQUFPLENBQUMsTUFBTTs0QkFDZCxTQUFTO3lCQUNWO3dCQUNELEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7d0JBQ2xDLHdCQUF3QixFQUFFLFVBQVUsS0FBSyxPQUFPO3FCQUNqRCxDQUNGLENBQUM7b0JBQ0YsT0FBTyxJQUFJLENBQUM7Z0JBQ2QsQ0FBQzthQUNGLENBQUM7U0FDSDtJQUNILENBQUM7SUFsR00sTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFzQjs7UUFDekMsTUFBTSxRQUFRLEdBQUcsSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFdkMsT0FBTyxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUNyRCxhQUFhLFFBQUUsT0FBTyxDQUFDLGFBQWEsbUNBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ2hFLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUztZQUM1QixRQUFRLEVBQUU7Z0JBQ1IsS0FBSyxFQUFFLFFBQVEsQ0FBQyxLQUFLO2dCQUNyQixPQUFPLEVBQUUsUUFBUSxDQUFDLE9BQU87Z0JBQ3pCLFdBQVcsRUFBRSxRQUFRLENBQUMsV0FBVztnQkFDakMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxLQUFLO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLE1BQU0sQ0FBQyxxQkFBcUI7UUFDakMsSUFBSSxDQUFDLFdBQVcsR0FBRyxTQUFTLENBQUM7SUFDL0IsQ0FBQztJQW1GTSxxQkFBcUIsQ0FBQyxRQUFnQixFQUFFLFNBQWlCLEVBQUUsYUFBOEIsT0FBTzs7UUFDckcsTUFBTSxRQUFRLEdBQUcsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXhDLE1BQU0sU0FBUyxHQUFHLGFBQU0sQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFFbkUsTUFBTSxjQUFjLEdBQVc7WUFDN0IsSUFBSSxFQUFFLE9BQU87WUFDYixTQUFTLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQSxDQUFDLENBQUMsRUFBRTtZQUM3QixJQUFJLEVBQUUsR0FBRyxRQUFRLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxFQUFFO1lBQzNDLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFO1lBQ3JFLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLEVBQUU7U0FDaEQsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRTdCLE9BQU8sS0FBSyxDQUFDO1lBQ1gsZUFBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsY0FBYyxDQUFDLFFBQVEsRUFBRSxTQUFTLG9DQUFLLEVBQUU7WUFDckUsY0FBYztZQUNkLGVBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLDBDQUFFLGFBQWEsQ0FBQyxRQUFRLEVBQUUsU0FBUyxvQ0FBSyxFQUFFO1NBQ3JFLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQXpIRCw0QkF5SEM7QUFFRDs7R0FFRztBQUNILFNBQVMsVUFBVSxDQUFDLFFBQXlCO0lBQzNDLE9BQU8sVUFBUyxHQUFHLEtBQWU7UUFDaEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDO1FBQ25DLGdEQUFnRDtRQUNoRCxJQUFJLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxPQUFPLElBQUksUUFBUSxLQUFLLE9BQU8sRUFBRTtZQUNyRCxPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1NBQ25DO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQyxDQUFDO0FBQ0osQ0FBQztBQUVELFNBQVMsS0FBSyxDQUFDLFFBQWtCO0lBQy9CLE9BQU8sUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7QUFDaEQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBcmNoaXRlY3R1cmUsIEFzc2V0Q29kZSwgQ29kZSwgUnVudGltZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQnVuZGxpbmdPcHRpb25zIH0gZnJvbSAnLi90eXBlcyc7XG5pbXBvcnQgeyBleGVjLCBmaW5kVXAsIGdldEdvQnVpbGRWZXJzaW9uIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBPcHRpb25zIGZvciBidW5kbGluZ1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1bmRsaW5nUHJvcHMgZXh0ZW5kcyBCdW5kbGluZ09wdGlvbnMge1xuICAvKipcbiAgICogRGlyZWN0b3J5IGNvbnRhaW5pbmcgeW91ciBnby5tb2QgZmlsZVxuICAgKlxuICAgKiBUaGlzIHdpbGwgYWNjZXB0IGVpdGhlciBhIGRpcmVjdG9yeSBwYXRoIGNvbnRhaW5pbmcgYSBgZ28ubW9kYCBmaWxlXG4gICAqIG9yIGEgZmlsZXBhdGggdG8geW91ciBgZ28ubW9kYCBmaWxlIChpLmUuIGBwYXRoL3RvL2dvLm1vZGApLlxuICAgKlxuICAgKiBUaGlzIHdpbGwgYmUgdXNlZCBhcyB0aGUgc291cmNlIG9mIHRoZSB2b2x1bWUgbW91bnRlZCBpbiB0aGUgRG9ja2VyXG4gICAqIGNvbnRhaW5lciBhbmQgd2lsbCBiZSB0aGUgZGlyZWN0b3J5IHdoZXJlIGl0IHdpbGwgcnVuIGBnbyBidWlsZGAgZnJvbS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgcGF0aCBpcyBmb3VuZCBieSB3YWxraW5nIHVwIHBhcmVudCBkaXJlY3RvcmllcyBzZWFyY2hpbmcgZm9yXG4gICAqICBhIGBnby5tb2RgIGZpbGUgZnJvbSB0aGUgbG9jYXRpb24gb2YgYGVudHJ5YFxuICAgKi9cbiAgcmVhZG9ubHkgbW9kdWxlRGlyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXRoIHRvIHRoZSBmb2xkZXIgb3IgZmlsZSB0aGF0IGNvbnRhaW5zIHRoZSBtYWluIGFwcGxpY2F0aW9uIGVudHJ5IHBvaW50IGZpbGVzIGZvciB0aGUgcHJvamVjdC5cbiAgICpcbiAgICogVGhpcyBhY2NlcHRzIGVpdGhlciBhIHBhdGggdG8gYSBkaXJlY3Rvcnkgb3IgZmlsZS5cbiAgICpcbiAgICogSWYgYSBkaXJlY3RvcnkgcGF0aCBpcyBwcm92aWRlZCB0aGVuIGl0IHdpbGwgYXNzdW1lIHRoZXJlIGlzIGEgR28gZW50cnkgZmlsZSAoaS5lLiBgbWFpbi5nb2ApIGFuZFxuICAgKiB3aWxsIGNvbnN0cnVjdCB0aGUgYnVpbGQgY29tbWFuZCB1c2luZyB0aGUgZGlyZWN0b3J5IHBhdGguXG4gICAqXG4gICAqIEZvciBleGFtcGxlLCBpZiB5b3UgcHJvdmlkZSB0aGUgZW50cnkgYXM6XG4gICAqXG4gICAqICAgICBlbnRyeTogJ215LWxhbWJkYS1hcHAvY21kL2FwaSdcbiAgICpcbiAgICogVGhlbiB0aGUgYGdvIGJ1aWxkYCBjb21tYW5kIHdvdWxkIGJlOlxuICAgKlxuICAgKiAgICAgYGdvIGJ1aWxkIC4vY21kL2FwaWBcbiAgICpcbiAgICogSWYgYSBwYXRoIHRvIGEgZmlsZSBpcyBwcm92aWRlZCB0aGVuIGl0IHdpbGwgdXNlIHRoZSBmaWxlcGF0aCBpbiB0aGUgYnVpbGQgY29tbWFuZC5cbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIGlmIHlvdSBwcm92aWRlIHRoZSBlbnRyeSBhczpcbiAgICpcbiAgICogICAgIGVudHJ5OiAnbXktbGFtYmRhLWFwcC9jbWQvYXBpL21haW4uZ28nXG4gICAqXG4gICAqIFRoZW4gdGhlIGBnbyBidWlsZGAgY29tbWFuZCB3b3VsZCBiZTpcbiAgICpcbiAgICogICAgIGBnbyBidWlsZCAuL2NtZC9hcGkvbWFpbi5nb2BcbiAgICovXG4gIHJlYWRvbmx5IGVudHJ5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBydW50aW1lIG9mIHRoZSBsYW1iZGEgZnVuY3Rpb25cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU6IFJ1bnRpbWU7XG5cbiAgLyoqXG4gICAqIFRoZSBzeXN0ZW0gYXJjaGl0ZWN0dXJlIG9mIHRoZSBsYW1iZGEgZnVuY3Rpb25cbiAgICovXG4gIHJlYWRvbmx5IGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlO1xufVxuXG4vKipcbiAqIEJ1bmRsaW5nXG4gKi9cbmV4cG9ydCBjbGFzcyBCdW5kbGluZyBpbXBsZW1lbnRzIGNkay5CdW5kbGluZ09wdGlvbnMge1xuXG4gIHB1YmxpYyBzdGF0aWMgYnVuZGxlKG9wdGlvbnM6IEJ1bmRsaW5nUHJvcHMpOiBBc3NldENvZGUge1xuICAgIGNvbnN0IGJ1bmRsaW5nID0gbmV3IEJ1bmRsaW5nKG9wdGlvbnMpO1xuXG4gICAgcmV0dXJuIENvZGUuZnJvbUFzc2V0KHBhdGguZGlybmFtZShvcHRpb25zLm1vZHVsZURpciksIHtcbiAgICAgIGFzc2V0SGFzaFR5cGU6IG9wdGlvbnMuYXNzZXRIYXNoVHlwZSA/PyBjZGsuQXNzZXRIYXNoVHlwZS5PVVRQVVQsXG4gICAgICBhc3NldEhhc2g6IG9wdGlvbnMuYXNzZXRIYXNoLFxuICAgICAgYnVuZGxpbmc6IHtcbiAgICAgICAgaW1hZ2U6IGJ1bmRsaW5nLmltYWdlLFxuICAgICAgICBjb21tYW5kOiBidW5kbGluZy5jb21tYW5kLFxuICAgICAgICBlbnZpcm9ubWVudDogYnVuZGxpbmcuZW52aXJvbm1lbnQsXG4gICAgICAgIGxvY2FsOiBidW5kbGluZy5sb2NhbCxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGNsZWFyUnVuc0xvY2FsbHlDYWNoZSgpOiB2b2lkIHsgLy8gZm9yIHRlc3RzXG4gICAgdGhpcy5ydW5zTG9jYWxseSA9IHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIHJ1bnNMb2NhbGx5PzogYm9vbGVhbjtcblxuICAvLyBDb3JlIGJ1bmRsaW5nIG9wdGlvbnNcbiAgcHVibGljIHJlYWRvbmx5IGltYWdlOiBjZGsuRG9ja2VySW1hZ2U7XG4gIHB1YmxpYyByZWFkb25seSBjb21tYW5kOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcHVibGljIHJlYWRvbmx5IGxvY2FsPzogY2RrLklMb2NhbEJ1bmRsaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcmVsYXRpdmVFbnRyeVBhdGg6IHN0cmluZztcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBCdW5kbGluZ1Byb3BzKSB7XG4gICAgQnVuZGxpbmcucnVuc0xvY2FsbHkgPSBCdW5kbGluZy5ydW5zTG9jYWxseVxuICAgICAgPz8gZ2V0R29CdWlsZFZlcnNpb24oKVxuICAgICAgPz8gZmFsc2U7XG5cbiAgICBjb25zdCBwcm9qZWN0Um9vdCA9IHBhdGguZGlybmFtZShwcm9wcy5tb2R1bGVEaXIpO1xuICAgIHRoaXMucmVsYXRpdmVFbnRyeVBhdGggPSBgLi8ke3BhdGgucmVsYXRpdmUocHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShwcm9wcy5lbnRyeSkpfWA7XG5cbiAgICBjb25zdCBjZ29FbmFibGVkID0gcHJvcHMuY2dvRW5hYmxlZCA/ICcxJyA6ICcwJztcblxuICAgIGNvbnN0IGVudmlyb25tZW50ID0ge1xuICAgICAgQ0dPX0VOQUJMRUQ6IGNnb0VuYWJsZWQsXG4gICAgICBHTzExMU1PRFVMRTogJ29uJyxcbiAgICAgIEdPQVJDSDogcHJvcHMuYXJjaGl0ZWN0dXJlLmRvY2tlclBsYXRmb3JtLnNwbGl0KCcvJylbMV0sXG4gICAgICBHT09TOiAnbGludXgnLFxuICAgICAgLi4ucHJvcHMuZW52aXJvbm1lbnQsXG4gICAgfTtcblxuICAgIC8vIERvY2tlciBidW5kbGluZ1xuICAgIGNvbnN0IHNob3VsZEJ1aWxkSW1hZ2UgPSBwcm9wcy5mb3JjZWREb2NrZXJCdW5kbGluZyB8fCAhQnVuZGxpbmcucnVuc0xvY2FsbHk7XG4gICAgdGhpcy5pbWFnZSA9IHNob3VsZEJ1aWxkSW1hZ2VcbiAgICAgID8gcHJvcHMuZG9ja2VySW1hZ2UgPz8gY2RrLkRvY2tlckltYWdlLmZyb21CdWlsZChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGliJyksIHtcbiAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgLi4ucHJvcHMuYnVpbGRBcmdzID8/IHt9LFxuICAgICAgICAgIElNQUdFOiBSdW50aW1lLkdPXzFfWC5idW5kbGluZ0ltYWdlLmltYWdlLCAvLyBhbHdheXMgdXNlIHRoZSBHT18xX1ggYnVpbGQgaW1hZ2VcbiAgICAgICAgfSxcbiAgICAgICAgcGxhdGZvcm06IHByb3BzLmFyY2hpdGVjdHVyZS5kb2NrZXJQbGF0Zm9ybSxcbiAgICAgIH0pXG4gICAgICA6IGNkay5Eb2NrZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ2R1bW15Jyk7IC8vIERvIG5vdCBidWlsZCBpZiB3ZSBkb24ndCBuZWVkIHRvXG5cbiAgICBjb25zdCBidW5kbGluZ0NvbW1hbmQgPSB0aGlzLmNyZWF0ZUJ1bmRsaW5nQ29tbWFuZChjZGsuQXNzZXRTdGFnaW5nLkJVTkRMSU5HX0lOUFVUX0RJUiwgY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19PVVRQVVRfRElSKTtcbiAgICB0aGlzLmNvbW1hbmQgPSBbJ2Jhc2gnLCAnLWMnLCBidW5kbGluZ0NvbW1hbmRdO1xuICAgIHRoaXMuZW52aXJvbm1lbnQgPSBlbnZpcm9ubWVudDtcblxuICAgIC8vIExvY2FsIGJ1bmRsaW5nXG4gICAgaWYgKCFwcm9wcy5mb3JjZWREb2NrZXJCdW5kbGluZykgeyAvLyBvbmx5IGlmIERvY2tlciBpcyBub3QgZm9yY2VkXG5cbiAgICAgIGNvbnN0IG9zUGxhdGZvcm0gPSBvcy5wbGF0Zm9ybSgpO1xuICAgICAgY29uc3QgY3JlYXRlTG9jYWxDb21tYW5kID0gKG91dHB1dERpcjogc3RyaW5nKSA9PiB0aGlzLmNyZWF0ZUJ1bmRsaW5nQ29tbWFuZChwcm9qZWN0Um9vdCwgb3V0cHV0RGlyLCBvc1BsYXRmb3JtKTtcblxuICAgICAgdGhpcy5sb2NhbCA9IHtcbiAgICAgICAgdHJ5QnVuZGxlKG91dHB1dERpcjogc3RyaW5nKSB7XG4gICAgICAgICAgaWYgKEJ1bmRsaW5nLnJ1bnNMb2NhbGx5ID09IGZhbHNlKSB7XG4gICAgICAgICAgICBwcm9jZXNzLnN0ZGVyci53cml0ZSgnZ28gYnVpbGQgY2Fubm90IHJ1biBsb2NhbGx5LiBTd2l0Y2hpbmcgdG8gRG9ja2VyIGJ1bmRsaW5nLlxcbicpO1xuICAgICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGNvbnN0IGxvY2FsQ29tbWFuZCA9IGNyZWF0ZUxvY2FsQ29tbWFuZChvdXRwdXREaXIpO1xuICAgICAgICAgIGV4ZWMoXG4gICAgICAgICAgICBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ2NtZCcgOiAnYmFzaCcsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIG9zUGxhdGZvcm0gPT09ICd3aW4zMicgPyAnL2MnIDogJy1jJyxcbiAgICAgICAgICAgICAgbG9jYWxDb21tYW5kLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgZW52OiB7IC4uLnByb2Nlc3MuZW52LCAuLi5lbnZpcm9ubWVudCA/PyB7fSB9LFxuICAgICAgICAgICAgICBzdGRpbzogWyAvLyBzaG93IG91dHB1dFxuICAgICAgICAgICAgICAgICdpZ25vcmUnLCAvLyBpZ25vcmUgc3RkaW9cbiAgICAgICAgICAgICAgICBwcm9jZXNzLnN0ZGVyciwgLy8gcmVkaXJlY3Qgc3Rkb3V0IHRvIHN0ZGVyclxuICAgICAgICAgICAgICAgICdpbmhlcml0JywgLy8gaW5oZXJpdCBzdGRlcnJcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgY3dkOiBwYXRoLmRpcm5hbWUocHJvcHMubW9kdWxlRGlyKSxcbiAgICAgICAgICAgICAgd2luZG93c1ZlcmJhdGltQXJndW1lbnRzOiBvc1BsYXRmb3JtID09PSAnd2luMzInLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICApO1xuICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgY3JlYXRlQnVuZGxpbmdDb21tYW5kKGlucHV0RGlyOiBzdHJpbmcsIG91dHB1dERpcjogc3RyaW5nLCBvc1BsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0gPSAnbGludXgnKTogc3RyaW5nIHtcbiAgICBjb25zdCBwYXRoSm9pbiA9IG9zUGF0aEpvaW4ob3NQbGF0Zm9ybSk7XG5cbiAgICBjb25zdCBoYXNWZW5kb3IgPSBmaW5kVXAoJ3ZlbmRvcicsIHBhdGguZGlybmFtZSh0aGlzLnByb3BzLmVudHJ5KSk7XG5cbiAgICBjb25zdCBnb0J1aWxkQ29tbWFuZDogc3RyaW5nID0gW1xuICAgICAgJ2dvJywgJ2J1aWxkJyxcbiAgICAgIGhhc1ZlbmRvciA/ICctbW9kPXZlbmRvcic6ICcnLFxuICAgICAgJy1vJywgYCR7cGF0aEpvaW4ob3V0cHV0RGlyLCAnYm9vdHN0cmFwJyl9YCxcbiAgICAgIGAke3RoaXMucHJvcHMuZ29CdWlsZEZsYWdzID8gdGhpcy5wcm9wcy5nb0J1aWxkRmxhZ3Muam9pbignICcpIDogJyd9YCxcbiAgICAgIGAke3RoaXMucmVsYXRpdmVFbnRyeVBhdGgucmVwbGFjZSgvXFxcXC9nLCAnLycpfWAsXG4gICAgXS5maWx0ZXIoYyA9PiAhIWMpLmpvaW4oJyAnKTtcblxuICAgIHJldHVybiBjaGFpbihbXG4gICAgICAuLi50aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYmVmb3JlQnVuZGxpbmcoaW5wdXREaXIsIG91dHB1dERpcikgPz8gW10sXG4gICAgICBnb0J1aWxkQ29tbWFuZCxcbiAgICAgIC4uLnRoaXMucHJvcHMuY29tbWFuZEhvb2tzPy5hZnRlckJ1bmRsaW5nKGlucHV0RGlyLCBvdXRwdXREaXIpID8/IFtdLFxuICAgIF0pO1xuICB9XG59XG5cbi8qKlxuICogUGxhdGZvcm0gc3BlY2lmaWMgcGF0aCBqb2luXG4gKi9cbmZ1bmN0aW9uIG9zUGF0aEpvaW4ocGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge1xuICByZXR1cm4gZnVuY3Rpb24oLi4ucGF0aHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgICBjb25zdCBqb2luZWQgPSBwYXRoLmpvaW4oLi4ucGF0aHMpO1xuICAgIC8vIElmIHdlIGFyZSBvbiB3aW4zMiBidXQgbmVlZCBwb3NpeCBzdHlsZSBwYXRoc1xuICAgIGlmIChvcy5wbGF0Zm9ybSgpID09PSAnd2luMzInICYmIHBsYXRmb3JtICE9PSAnd2luMzInKSB7XG4gICAgICByZXR1cm4gam9pbmVkLnJlcGxhY2UoL1xcXFwvZywgJy8nKTtcbiAgICB9XG4gICAgcmV0dXJuIGpvaW5lZDtcbiAgfTtcbn1cblxuZnVuY3Rpb24gY2hhaW4oY29tbWFuZHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgcmV0dXJuIGNvbW1hbmRzLmZpbHRlcihjID0+ICEhYykuam9pbignICYmICcpO1xufVxuIl19