"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3DeployAction = exports.CacheControl = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /** The 'must-revalidate' cache control directive. */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /** The 'no-cache' cache control directive. */
    static noCache() { return new CacheControl('no-cache'); }
    /** The 'no-transform' cache control directive. */
    static noTransform() { return new CacheControl('no-transform'); }
    /** The 'public' cache control directive. */
    static setPublic() { return new CacheControl('public'); }
    /** The 'private' cache control directive. */
    static setPrivate() { return new CacheControl('private'); }
    /** The 'proxy-revalidate' cache control directive. */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /** The 'max-age' cache control directive. */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /** The 's-max-age' cache control directive. */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive,
     * in case our support is missing a method for a particular directive.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 */
class S3DeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,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