"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsDeployAction = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline Action to deploy an ECS Service.
 */
class EcsDeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'ECS',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [determineInputArtifact(props)],
            resource: props.service,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // permissions based on CodePipeline documentation:
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/how-to-custom-role.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'ecs:DescribeTaskDefinition',
                'ecs:DescribeTasks',
                'ecs:ListTasks',
                'ecs:RegisterTaskDefinition',
                'ecs:UpdateService',
            ],
            resources: ['*'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        options.bucket.grantRead(options.role);
        return {
            configuration: {
                ClusterName: this.props.service.cluster.clusterName,
                ServiceName: this.props.service.serviceName,
                FileName: this.props.imageFile && this.props.imageFile.fileName,
            },
        };
    }
}
exports.EcsDeployAction = EcsDeployAction;
function determineInputArtifact(props) {
    if (props.imageFile && props.input) {
        throw new Error("Exactly one of 'input' or 'imageFile' can be provided in the ECS deploy Action");
    }
    if (props.imageFile) {
        return props.imageFile.artifact;
    }
    if (props.input) {
        return props.input;
    }
    throw new Error("Specifying one of 'input' or 'imageFile' is required for the ECS deploy Action");
}
//# sourceMappingURL=data:application/json;base64,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