"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BitBucketSourceAction = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * A CodePipeline source action for BitBucket.
 *
 * @experimental
 */
class BitBucketSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'AWS',
            provider: 'CodeStarSourceConnection',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        var _a;
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/security-iam.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codestar-connections:UseConnection',
            ],
            resources: [
                this.props.connectionArn,
            ],
        }));
        // the action needs to write the output to the pipeline bucket
        options.bucket.grantReadWrite(options.role);
        // if codeBuildCloneOutput is true,
        // save the connectionArn in the Artifact instance
        // to be read by the CodeBuildAction later
        if (this.props.codeBuildCloneOutput === true) {
            this.props.output.setMetadata(BitBucketSourceAction._CONNECTION_ARN_PROPERTY, this.props.connectionArn);
        }
        return {
            configuration: {
                ConnectionArn: this.props.connectionArn,
                FullRepositoryId: `${this.props.owner}/${this.props.repo}`,
                BranchName: (_a = this.props.branch) !== null && _a !== void 0 ? _a : 'master',
                OutputArtifactFormat: this.props.codeBuildCloneOutput === true
                    ? 'CODEBUILD_CLONE_REF'
                    : undefined,
            },
        };
    }
}
exports.BitBucketSourceAction = BitBucketSourceAction;
/**
 * The name of the property that holds the ARN of the CodeStar Connection
 * inside of the CodePipeline Artifact's metadata.
 *
 * @internal
 */
BitBucketSourceAction._CONNECTION_ARN_PROPERTY = 'CodeStarConnectionArnProperty';
//# sourceMappingURL=data:application/json;base64,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