"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupSmithyBuild = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const smithy_build_project_1 = require("./smithy-build-project");
const smithy_model_project_1 = require("./smithy-model-project");
/**
 * Adds the common subprojects for building a Smithy model into an OpenAPI spec
 */
exports.setupSmithyBuild = (project, options) => {
    const modelDir = options.modelDir ?? "model";
    const fullyQualifiedServiceName = options.serviceName ?? "example.hello#Hello";
    const [serviceNamespace, serviceName] = fullyQualifiedServiceName.split("#");
    const smithyBuildDir = "smithy-build";
    // Create a smithy model (eg main.smithy)
    const smithyModel = new smithy_model_project_1.SmithyModelProject({
        name: `${project.name}-smithy`,
        outdir: modelDir,
        parent: project,
        serviceNamespace,
        serviceName,
    });
    smithyModel.synth();
    const smithyBuildOutputSubDir = "output";
    // Create the smithy build project, responsible for transforming the model into an OpenAPI spec
    const smithyBuild = new smithy_build_project_1.SmithyBuildProject({
        name: `${project.name}-smithy-build`,
        parent: project,
        outdir: smithyBuildDir,
        fullyQualifiedServiceName,
        smithyBuildOptions: options.smithyBuildOptions,
        modelPath: path.join(project.outdir, modelDir),
        buildOutputDir: smithyBuildOutputSubDir,
        gradleWrapperPath: options.gradleWrapperPath
            ? path.resolve(project.outdir, options.gradleWrapperPath)
            : undefined,
    });
    smithyBuild.synth();
    const smithyBuildOutputDir = path.join(smithyBuildDir, smithyBuildOutputSubDir);
    // Ignore smithy build output by default
    if (options.ignoreSmithyBuildOutput ?? true) {
        project.gitignore.addPatterns(smithyBuildOutputDir);
    }
    // Ignore the .gradle directory
    project.gitignore.addPatterns(path.join(smithyBuildDir, ".gradle"));
    return {
        modelDir,
        // Generated spec is written to output/<projection id>/<plugin id>/<service name>.openapi.json
        generatedSpecFilePath: path.join(smithyBuild.smithyBuildOutputPath, "openapi", "openapi", `${serviceName}.openapi.json`),
    };
};
//# sourceMappingURL=data:application/json;base64,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