"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const java_2 = require("./samples/java");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "software.aws.awsprototypingsdk/open-api-gateway@^0";
/**
 * Synthesizes a Java Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-java
 */
class OpenApiGatewayJavaProject extends java_1.JavaProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sample: false,
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "software.constructs/constructs@^10",
            "software.amazon.awscdk/aws-cdk-lib@^2",
            "io.github.cdklabs/cdknag@^2",
            "org.projectlombok/lombok@^1",
            "com.fasterxml.jackson.core/jackson-databind@^2",
            "io.github.cdklabs/projen@^0",
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Synthesize the openapi spec early since it's used by the generated java client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join("src", this.specDir),
            // Write the parsed spec to the resources directory so that it can be packaged into the jar
            parsedSpecOutdir: path.join(...this.specDir.split("/").map(() => ".."), "main", "resources", this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the java client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.JAVA);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.JAVA,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedJavaClient = this.generatedClients[languages_1.ClientLanguage.JAVA];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedJavaClient.synth();
        // Add a dependency on the generated java client
        this.addDependency(`${this.generatedJavaClient.pom.groupId}/${this.generatedJavaClient.pom.artifactId}@${this.generatedJavaClient.pom.version}`);
        // Add a dependency on the generated java client repository
        this.pom.addRepository({
            url: `file://./${this.generatedCodeDir}/${languages_1.ClientLanguage.JAVA}/dist/java`,
            id: "generated-java-api-client",
        });
        if (this.hasParent) {
            // Since the generated java client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated java client is built first
                this.parent.addImplicitDependency(this, this.generatedJavaClient);
            }
        }
        // We build this project as a super jar so that it can be deployed as a lambda. This isn't ideal for jar size and
        // so is not the recommended approach, however is makes it possible for a "one-click" way to get started with a
        // full hello world api. Included in the generated "SampleApi.java" is a comment encouraging users to read the
        // README and to move to defining a separate package for lambda handlers.
        this.pom.addPlugin("org.apache.maven.plugins/maven-shade-plugin@3.3.0", {
            configuration: {
                createDependencyReducedPom: false,
                // Name is the same as the regular maven build jar, which ensures the api jar path resolves to the super jar
                // for deployment as a lambda.
                finalName: `${this.name}-${options.version}`,
            },
            executions: [
                {
                    id: "shade-task",
                    phase: "package",
                    goals: ["shade"],
                },
            ],
        });
        const javaClientPackage = `${this.generatedJavaClient.pom.groupId}.${this.generatedJavaClient.name}.client`;
        // Generate the sample source code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            sampleCode: options.sample ?? true,
            apiSrcDir: this.apiSrcDir,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            javaClientPackage,
        };
        new projen_1.SampleDir(this, path.join("src", "main", "java", this.apiSrcDir), {
            files: java_2.getJavaSampleSource(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than java project options.
     */
    preConstruct(options) {
        return options;
    }
}
exports.OpenApiGatewayJavaProject = OpenApiGatewayJavaProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayJavaProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayJavaProject", version: "0.11.5" };
//# sourceMappingURL=data:application/json;base64,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