import { Construct } from 'constructs';
import { StateMachine } from 'aws-cdk-lib/aws-stepfunctions';
import { EventBus } from 'aws-cdk-lib/aws-events';
import { LfAccessControlMode } from './central-governance';
import { DataLakeStorage } from '../data-lake-storage';
/**
 * Properties for the DataDomain Construct
 */
export interface DataDomainProps {
    /**
    * Data domain name
    */
    readonly domainName: string;
    /**
    * Central Governance account Id
    */
    readonly centralAccountId: string;
    /**
    * Flag to create a Crawler workflow in Data Domain account
    */
    readonly crawlerWorkflow?: boolean;
}
/**
 * This CDK Construct creates all required resources for data mesh in Data Domain account.
 *
 * It creates the following:
 * * A data lake with multiple layers (Raw, Cleaned, Transformed) using {@link DataLakeStorage} construct
 * * An mazon EventBridge Event Bus and Rules to enable Central Governance account to send events to Data Domain account
 * * An AWS Secret Manager secret encrypted via AWS KMS and used to share references with the central governance account
 * * A Data Domain Workflow {@link DataDomainWorkflow} responsible for creating resources in the data domain via a Step Functions state machine
 * * An optional Crawler workflow {@link DataDomainCrawler} responsible for updating the data product schema after registration via a Step Functions state machine
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * new DataDomain(stack, 'myDataDomain', {
 *  centralAccountId: '1234567891011',
 *  crawlerWorkflow: true,
 *  domainName: 'domainName'
 * });
 * ```
 */
export declare class DataDomain extends Construct {
    static readonly DATA_PRODUCTS_PREFIX: string;
    static readonly DOMAIN_CONFIG_SECRET: string;
    static readonly DOMAIN_BUS_NAME: string;
    readonly centralAccountId: string;
    readonly eventBus: EventBus;
    readonly dataLake: DataLakeStorage;
    /**
     * Construct a new instance of DataDomain.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainProps} props the DataDomainProps properties
     * @access public
     */
    constructor(scope: Construct, id: string, props: DataDomainProps);
    addBusRule(id: string, mode: LfAccessControlMode, workflow: StateMachine): void;
}
