import { Construct } from 'constructs';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { IEventBus } from 'aws-cdk-lib/aws-events';
/**
 * Enum to define access control mode in Lake Formation
 */
export declare enum LfAccessControlMode {
    NRAC = "nrac",
    TBAC = "tbac"
}
/**
 * LF Tag interface
 */
export interface LfTag {
    readonly key: string;
    readonly values: string[];
}
/**
 * Properties for the CentralGovernance Construct
 */
export interface CentralGovernanceProps {
    /**
     * LF tags
     */
    readonly lfTags?: LfTag[];
}
/**
 * This CDK Construct creates a Data Product registration workflow and resources for the Central Governance account.
 * It uses AWS Step Functions state machine to orchestrate the workflow:
 * * creates tables in AWS Glue Data Catalog
 * * shares tables to Data Product owner account (Producer)
 *
 * This construct also creates an Amazon EventBridge Event Bus to enable communication with Data Domain accounts (Producer/Consumer).
 *
 * This construct requires to use the default [CDK qualifier](https://docs.aws.amazon.com/cdk/v2/guide/bootstrapping.html) generated with the standard CDK bootstrap stack.
 * It ensures the right CDK execution role is used and granted Lake Formation administrator permissions so CDK can create Glue databases when registring a DataDomain.
 *
 * To register a DataDomain, the following information are required:
 * * The account Id of the DataDomain
 * * The secret ARN for the domain configuration available as a CloudFormation output when creating a {@link DataDomain}
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { CentralGovernance, LfTag } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'CentralGovStack');
 *
 * const tags: LfTag[] = [{key: 'tag1': values:['LfTagValue1', 'LfTagValue2']}]
 * const governance = new CentralGovernance(stack, 'myCentralGov', { tags });
 *
 * governance.registerDataDomain('Domain1', 'domain1Name', <DOMAIN_CONFIG_SECRET_ARN>);
 * ```
 */
export declare class CentralGovernance extends Construct {
    static readonly CENTRAL_BUS_NAME: string;
    static readonly DOMAIN_DATABASE_PREFIX: string;
    static readonly DOMAIN_TAG_KEY: string;
    readonly workflowRole: IRole;
    readonly eventBus: IEventBus;
    private readonly cdkLfAdmin;
    private lfTags;
    /**
     * Construct a new instance of CentralGovernance.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CentralGovernanceProps} props the CentralGovernance properties
     * @access public
     */
    constructor(scope: Construct, id: string, props?: CentralGovernanceProps);
    /**
     * Registers a new Data Domain account in Central Governance account.
     * Each Data Domain account {@link DataDomain} has to be registered in Central Gov. account before it can participate in a mesh.
     *
     * It creates:
     * * A cross-account policy for Amazon EventBridge Event Bus to enable Data Domain to send events to Central Gov. account
     * * A Lake Formation data access role scoped down to the data domain products bucket
     * * A Glue Catalog Database to hold Data Products for this Data Domain
     * * A Rule to forward events to target Data Domain account.
     *
     * Object references are passed from the DataDomain account to the CentralGovernance account via a AWS Secret Manager secret and cross account access.
     * It includes the following JSON object:
     * ```json
     * {
     *   BucketName: 'clean-<ACCOUNT_ID>-<REGION>',
     *   Prefix: 'data-products',
     *   KmsKeyId: '<KMS_ID>,
     * }
     * ```
     *
     * @param {string} id the ID of the CDK Construct
     * @param {string} domainId the account ID of the DataDomain to register
     * @param {string} domainName the name of the DataDomain, i.e. Line of Business name
     * @param {string} domainSecretArn the full ARN of the secret used by producers to share references with the central governance
     * @param {LfAccessControlMode} lfAccessControlMode Lake Formation Access Control mode for the DataDomain
     * @access public
     */
    registerDataDomain(id: string, domainId: string, domainName: string, domainSecretArn: string, lfAccessControlMode?: LfAccessControlMode): void;
}
