"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkDeployer = exports.DeploymentType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const cdk = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const singleton_kms_key_1 = require("../singleton-kms-key");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const utils_1 = require("../utils");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const cdk_deployer_build_1 = require("./cdk-deployer-build");
var DeploymentType;
(function (DeploymentType) {
    DeploymentType[DeploymentType["WORKSHOP_STUDIO"] = 0] = "WORKSHOP_STUDIO";
    DeploymentType[DeploymentType["CLICK_TO_DEPLOY"] = 1] = "CLICK_TO_DEPLOY";
})(DeploymentType = exports.DeploymentType || (exports.DeploymentType = {}));
/**
 * A custom CDK Stack that can be synthetized as a CloudFormation Stack to deploy a CDK application hosted on GitHub or on S3 as a Zip file.
 * This stack is self contained and can be one-click deployed to any AWS account.
 * It can be used for AWS workshop or AWS blog examples deployment when CDK is not supported/desired.
 * The stack supports passing the CDK application stack name to deploy (in case there are multiple stacks in the CDK app) and CDK parameters.
 *
 * It contains the necessary resources to synchronously deploy a CDK application from a GitHub repository:
 *  * A CodeBuild project to effectively deploy the CDK application
 *  * A StartBuild custom resource to synchronously triggers the build using a callback pattern based on Event Bridge
 *  * The necessary roles and permissions
 *
 * The StartBuild CFN custom resource is using the callback pattern to wait for the build completion:
 *  1. a Lambda function starts the build but doesn't return any value to the CFN callback URL. Instead, the callback URL is passed to the build project.
 *  2. the completion of the build triggers an Event and a second Lambda function which checks the result of the build and send information to the CFN callback URL
 *
 *  * Usage example:
 * ```typescript
 * new CdkDeployer(AwsNativeRefArchApp, 'AwsNativeRefArchDeployer', {
 *  githubRepository: 'aws-samples/aws-analytics-reference-architecture',
 *  cdkAppLocation: 'refarch/aws-native',
 *  cdkParameters: {
 *    QuickSightUsername: {
 *      default: 'myuser',
 *      type: 'String',
 *    },
 *    QuickSightIdentityRegion: {
 *      default: 'us-east-1',
 *      type: 'String',
 *    },
 *  },
 * });
 * ```
 */
class CdkDeployer extends cdk.Stack {
    /**
     * Constructs a new instance of the TrackedConstruct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CdkDeployerProps} props the CdkDeployer [properties]{@link CdkDeployerProps}
     */
    constructor(scope, props) {
        super(scope, 'CDKDeployer', {
            // Change the Stack Synthetizer to remove the CFN parameters for the CDK version
            synthesizer: new aws_cdk_lib_1.DefaultStackSynthesizer({
                generateBootstrapVersionRule: false,
            }),
        });
        // Add parameters to the stack so it can be transfered to the CDK application
        var parameters = '';
        for (let name in props.cdkParameters) {
            let param = props.cdkParameters[name];
            let cfnParam = new cdk.CfnParameter(this, name, param);
            parameters = parameters.concat(` -c ${name}=${cfnParam.value}`);
        }
        // Name of the stack to deploy in codebuild
        const stackName = props.stackName ? props.stackName : '';
        // Role used by the CodeBuild project
        const buildRole = new aws_iam_1.Role(this, 'CodeBuildRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('codebuild.amazonaws.com'),
        });
        // We need the CDK execution role so the CodeBuild role can assume it for CDK deployment
        const cdkDeployRole = utils_1.Utils.getCdkDeployRole(this, 'CdkDeployRole');
        const cdkPublishRole = utils_1.Utils.getCdkFilePublishRole(this, 'CdkPublishRole');
        buildRole.addManagedPolicy(new aws_iam_1.ManagedPolicy(this, 'CdkBuildPolicy', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'kms:CreateKey',
                        'kms:DisableKey',
                        'kms:EnableKeyRotation',
                        'kms:TagResource',
                        'kms:DescribeKey',
                        'kms:ScheduleKeyDeletion',
                        'kms:CreateAlias',
                        'kms:DeleteAlias',
                        'kms:CreateGrant',
                        'kms:DescribeKey',
                        'kms:RetireGrant'
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        's3:CreateBucket',
                        's3:PutBucketAcl',
                        's3:PutEncryptionConfiguration',
                        's3:PutBucketPublicAccessBlock',
                        's3:PutBucketVersioning',
                        's3:DeleteBucket',
                        's3:PutBucketPolicy',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:cloudformation:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:stack/CDKToolkit*`
                    ],
                    actions: [
                        'cloudformation:DescribeStacks',
                        'cloudformation:DeleteStack',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:ExecuteChangeSet',
                        'cloudformation:DescribeStackEvents',
                        'cloudformation:GetTemplate',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        cdkDeployRole.roleArn,
                        cdkPublishRole.roleArn,
                    ],
                    actions: [
                        'sts:AssumeRole',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:ssm:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:parameter/cdk-bootstrap/*/*`,
                    ],
                    actions: [
                        'ssm:PutParameter',
                        'ssm:GetParameters',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:ecr:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:repository/cdk*`,
                    ],
                    actions: [
                        'ecr:SetRepositoryPolicy',
                        'ecr:GetLifecyclePolicy',
                        'ecr:PutImageTagMutability',
                        'ecr:DescribeRepositories',
                        'ecr:ListTagsForResource',
                        'ecr:PutImageScanningConfiguration',
                        'ecr:CreateRepository',
                        'ecr:PutLifecyclePolicy',
                        'ecr:SetRepositoryPolicy',
                        'ecr:DeleteRepository',
                        'ecr:TagResource',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk*`,
                    ],
                    actions: [
                        'iam:GetRole',
                        'iam:CreateRole',
                        'iam:TagRole',
                        'iam:DeleteRole',
                        'iam:AttachRolePolicy',
                        'iam:DetachRolePolicy',
                        'iam:GetRolePolicy',
                        'iam:PutRolePolicy',
                        'iam:DeleteRolePolicy',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/codebuild/*`,
                    ],
                    actions: [
                        'logs:PutLogEvents',
                    ],
                }),
            ]
        }));
        let source;
        if (props.deploymentType === DeploymentType.WORKSHOP_STUDIO) {
            const cdkAppSourceCodeBucketName = new aws_cdk_lib_1.CfnParameter(this, 'CDKAppSourceCodeBucketName', {
                type: 'String',
            });
            const cdkAppSourceCodeBucketPrefix = new aws_cdk_lib_1.CfnParameter(this, 'CDKAppSourceCodeBucketPrefix', {
                type: 'String',
            });
            source = aws_codebuild_1.Source.s3({
                bucket: aws_s3_1.Bucket.fromBucketName(this, 'CdkAppBucket', cdkAppSourceCodeBucketName.valueAsString),
                path: `${cdkAppSourceCodeBucketPrefix.valueAsString}cdk_app.zip`,
            });
        }
        else {
            if (props.githubRepository) {
                source = aws_codebuild_1.Source.gitHub({
                    owner: props.githubRepository.split('/')[0],
                    repo: props.githubRepository.split('/')[1],
                    branchOrRef: props.gitBranch ? props.gitBranch : undefined,
                    reportBuildStatus: true,
                });
            }
            else if (props.s3Repository) {
                source = aws_codebuild_1.Source.s3({
                    bucket: aws_s3_1.Bucket.fromBucketName(this, 'CdkAppBucket', props.s3Repository.bucketName),
                    path: props.s3Repository.objectKey,
                });
            }
            else {
                throw new Error('githubRepository or s3Repository is required for CLICK_TO_DEPLOY deployment type');
            }
        }
        const codeBuildProject = new aws_codebuild_1.Project(this, 'CodeBuildProject', {
            source,
            encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(this, 'DefaultKmsKey'),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                computeType: aws_codebuild_1.ComputeType.SMALL,
                environmentVariables: {
                    PARAMETERS: {
                        value: parameters,
                    },
                    STACKNAME: {
                        value: stackName,
                    },
                    CDK_APP_LOCATION: {
                        value: props.cdkAppLocation ? props.cdkAppLocation : '',
                    },
                },
            },
            role: buildRole,
        });
        if (props.deploymentType === DeploymentType.WORKSHOP_STUDIO) {
            codeBuildProject.node.defaultChild.addPropertyOverride('EncryptionKey', 'alias/aws/s3');
        }
        const startBuildRole = new aws_iam_1.Role(this, 'StartBuildRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
            inlinePolicies: {
                StartBuild: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            resources: [codeBuildProject.projectArn],
                            actions: ['codebuild:StartBuild'],
                        }),
                    ],
                }),
            },
        });
        const startBuildFunction = new aws_lambda_1.Function(this, 'StartBuildFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromInline(cdk_deployer_build_1.startBuild(props.deployBuildSpec, props.destroyBuildSpec)),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(60),
            role: startBuildRole,
        });
        const reportBuildRole = new aws_iam_1.Role(this, 'ReportBuildRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
            inlinePolicies: {
                ReportBuild: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            resources: [codeBuildProject.projectArn],
                            actions: ['codebuild:BatchGetBuilds', 'codebuild:ListBuildsForProject'],
                        }),
                    ],
                }),
            },
        });
        const reportBuildFunction = new aws_lambda_1.Function(this, 'ReportBuildFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromInline(cdk_deployer_build_1.reportBuild),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(60),
            role: reportBuildRole,
        });
        const buildCompleteRule = new aws_events_1.Rule(this, 'BuildCompleteEvent', {
            eventPattern: {
                source: ['aws.codebuild'],
                detailType: ['CodeBuild Build State Change'],
                detail: {
                    'build-status': ['SUCCEEDED', 'FAILED', 'STOPPED'],
                    'project-name': [codeBuildProject.projectName],
                },
            },
            targets: [new aws_events_targets_1.LambdaFunction(reportBuildFunction)],
        });
        const buildTrigger = new aws_cdk_lib_1.CustomResource(this, 'CodeBuildTriggerCustomResource', {
            serviceToken: startBuildFunction.functionArn,
            properties: {
                ProjectName: codeBuildProject.projectName,
                BuildRoleArn: buildRole.roleArn,
                Parameters: parameters,
                StackName: stackName,
            },
        });
        buildTrigger.node.addDependency(buildCompleteRule);
        buildTrigger.node.addDependency(buildRole);
        this.deployResult = buildTrigger.getAttString('BuildStatus');
    }
}
exports.CdkDeployer = CdkDeployer;
_a = JSII_RTTI_SYMBOL_1;
CdkDeployer[_a] = { fqn: "aws-analytics-reference-architecture.CdkDeployer", version: "2.7.2" };
//# sourceMappingURL=data:application/json;base64,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