"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.concatMethodAndPath = void 0;
/**
 * Serialise a method and path into a single string
 */
exports.concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
/**
 * HTTP methods supported by Open API v3
 */
var HttpMethods;
(function (HttpMethods) {
    HttpMethods["GET"] = "get";
    HttpMethods["PUT"] = "put";
    HttpMethods["POST"] = "post";
    HttpMethods["DELETE"] = "delete";
    HttpMethods["OPTIONS"] = "options";
    HttpMethods["HEAD"] = "head";
    HttpMethods["PATCH"] = "patch";
    HttpMethods["TRACE"] = "trace";
})(HttpMethods || (HttpMethods = {}));
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 */
const applyMethodAuthorizer = (methodAuthorizer) => {
    if (methodAuthorizer) {
        return {
            security: [
                {
                    [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                },
            ],
        };
    }
    // NONE is specified via x-amazon-apigateway-auth
    return {
        "x-amazon-apigateway-auth": {
            type: "NONE",
        },
    };
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { methodAuthorizer, functionInvocationUri } = integrations[operationName];
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri: functionInvocationUri,
            passthroughBehavior: "WHEN_NO_MATCH",
        },
        ...applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        exports.concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[exports.concatMethodAndPath(methodAndPath)];
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: options.securitySchemes,
        },
    };
};
//# sourceMappingURL=data:application/json;base64,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