"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": [
        "export * from './api';",
        ...(options.sampleCode !== false ? ["export * from './sample-api';"] : []),
    ].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.ts": `import { OpenApiGatewayLambdaApi, OpenApiGatewayLambdaApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import spec from "../${options.specDir}/${options.parsedSpecFileName}";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayLambdaApiProps, "spec" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
export class Api extends OpenApiGatewayLambdaApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Authorizers } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      defaultAuthorizer: Authorizers.iam(),
      integrations: {
        sayHello: {
          function: new NodejsFunction(scope, "say-hello"),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "sample-api.say-hello.ts": `import { sayHelloHandler, ApiError } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler<ApiError>(async (input) => {
  return {
    statusCode: 200,
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.apiSrcDir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        function: new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        }),
      }]),
      ) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,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