"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupSmithyBuild = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const smithy_build_project_1 = require("./smithy-build-project");
const smithy_model_project_1 = require("./smithy-model-project");
/**
 * Adds the common subprojects for building a Smithy model into an OpenAPI spec
 */
exports.setupSmithyBuild = (project, options) => {
    const modelDir = options.modelDir ?? "model";
    const { namespace: serviceNamespace, serviceName } = options.serviceName;
    const fullyQualifiedServiceName = `${serviceNamespace}#${serviceName}`;
    const smithyBuildDir = "smithy-build";
    // Create a smithy model (eg main.smithy)
    const smithyModel = new smithy_model_project_1.SmithyModelProject({
        name: `${project.name}-smithy`,
        outdir: modelDir,
        parent: project,
        serviceNamespace,
        serviceName,
    });
    smithyModel.synth();
    const smithyBuildOutputSubDir = "output";
    // Create the smithy build project, responsible for transforming the model into an OpenAPI spec
    const smithyBuild = new smithy_build_project_1.SmithyBuildProject({
        name: `${project.name}-smithy-build`,
        parent: project,
        outdir: smithyBuildDir,
        fullyQualifiedServiceName,
        smithyBuildOptions: options.smithyBuildOptions,
        modelPath: path.join(project.outdir, modelDir),
        buildOutputDir: smithyBuildOutputSubDir,
        gradleWrapperPath: options.gradleWrapperPath
            ? path.resolve(project.outdir, options.gradleWrapperPath)
            : undefined,
    });
    smithyBuild.synth();
    const smithyBuildOutputDir = path.join(smithyBuildDir, smithyBuildOutputSubDir);
    // Ignore smithy build output by default
    if (options.ignoreSmithyBuildOutput ?? true) {
        project.gitignore.addPatterns(smithyBuildOutputDir);
    }
    // Ignore the .gradle directory
    project.gitignore.addPatterns(path.join(smithyBuildDir, ".gradle"));
    return {
        modelDir,
        // Generated spec is written to output/<projection id>/<plugin id>/<service name>.openapi.json
        generatedSpecFilePath: path.join(smithyBuild.smithyBuildOutputPath, "openapi", "openapi", `${serviceName}.openapi.json`),
    };
};
//# sourceMappingURL=data:application/json;base64,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