"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    ...options.tsconfig?.compilerOptions,
                },
            },
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Allow json files to be imported (for importing the parsed spec)
        this.tsconfig?.addInclude(`${this.srcdir}/**/*.json`);
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib");
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.srcdir, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Delete the lib directory prior to compilation to ensure latest parsed spec json file is copied
        this.preCompileTask.exec(`rm -rf ${this.libdir}`);
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.TYPESCRIPT);
        this.generatedClients = generate_1.generateClientProjects(clientLanguages, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.package.packageName,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: options.defaultReleaseBranch,
                packageManager: options.packageManager,
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedTypescriptClient = this.generatedClients[languages_1.ClientLanguage.TYPESCRIPT];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            apiSrcDir: path.join(this.srcdir, this.apiSrcDir),
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleFile(this, path.join(this.srcdir, "index.ts"), {
            contents: `export * from "./${this.apiSrcDir}";`,
        });
        new projen_1.SampleDir(this, path.join(this.srcdir, this.apiSrcDir), {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
        // Generate documentation if requested
        if ((options.documentationFormats ?? []).length > 0) {
            new docs_project_1.DocsProject({
                parent: this,
                outdir: path.join(this.generatedCodeDir, "documentation"),
                name: "docs",
                formats: [...new Set(options.documentationFormats)],
                specPath: spec.parsedSpecPath,
            });
        }
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.3.8" };
//# sourceMappingURL=data:application/json;base64,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