import torch
import torch.nn as nn
from typing import Iterable, Tuple, List, Dict, Optional, Any, Callable
from dataclasses import fields, Field
from .._base.base import BaseAutoencoder, BuildGuardMixin

class HyperAE(BuildGuardMixin, nn.Module):
    """
    Hypernetwork wrapper around a base autoencoder.

    This module:
      - wraps a `BaseAutoencoder` and uses a hypernetwork (2 layered MLP with ReLU) 
        to generate a subset of its parameters on a per-input basis
      - keeps the remaining parameters shared across the batch
    """

    _GUARDED = {"forward"}

    def __init__(
        self,
        base_ae: BaseAutoencoder,
        target_modules: Iterable[str] = ("encoder", "sampling_layer", "decoder"),
        hidden_dim: int = 256,
    ):
        """
        Initialize the HyperAE wrapper.

        Args:
            base_ae:
                The underlying autoencoder instance to be controlled by the hypernetwork.
            target_modules:
                Iterable of module name prefixes inside `base_ae` whose parameters
                will be generated by the hypernetwork (e.g. "encoder", "decoder").
            hidden_dim:
                Hidden dimension for the MLP hypernetwork.
        """
        super().__init__()

        self.base_ae = base_ae
        self.target_modules = tuple(target_modules)
        self.hidden_dim = hidden_dim

        # (param_name, original_shape, flat_start, flat_end)
        self._param_info: List[Tuple[str, torch.Size, int, int]] = []

        # Parameters that remain shared (usual trainable params).
        self._shared_param_dict: Dict[str, torch.Tensor] = {}

        # Total number of scalar parameters generated by the hypernetwork.
        self.total_generated_params: int = 0

        # Hypernetwork (built lazily in build()).
        self.hypernet: Optional[nn.Module] = None

        # Output type information inferred from a sample call in build().
        self._output_type: Optional[type] = None
        self._output_field_names: Optional[List[str]] = None

        # Cached ModelOutput field metadata (set in build()).
        self._output_fields: Optional[Tuple[Field, ...]] = None

        # Cached vmap metadata / function (set in build()).
        self._in_dims_params: Optional[Dict[str, Optional[int]]] = None
        self._vmapped_call: Optional[Callable[..., Any]] = None


    @torch.no_grad()
    def build(self, input_sample: torch.Tensor) -> None:
        """
        Build the hypernetwork and prepare parameter bookkeeping.

        This method:
          - builds the underlying `base_ae`
          - infers the output `ModelOutput` type and field names
          - splits `base_ae` parameters into generated vs shared sets
          - constructs the MLP hypernetwork that outputs all generated parameters
        Must be called once before using `forward` (enforced by `BuildGuardMixin`).
        """
        # Ensure the base autoencoder is built (and warmed up via its own build).
        self.base_ae.build(input_sample)

        # Inspect a sample output to record the output structure.
        sample_out = self.base_ae(input_sample)
        self._output_type = type(sample_out)
        self._output_fields = fields(sample_out)
        self._output_field_names = [f.name for f in self._output_fields]

        # Reset metadata containers.
        self._param_info.clear()

        # Walk over all base_ae parameters and mark them as generated or shared.
        flat_offset = 0
        shared_param_names = []
        for name, p in self.base_ae.named_parameters():
            if any(name.startswith(m + ".") for m in self.target_modules):
                # Parameters inside target modules are generated by the hypernet.
                numel = p.numel()
                self._param_info.append(
                    (name, p.shape, flat_offset, flat_offset + numel)
                )
                flat_offset += numel
                # Generated parameters are "owned" by the hypernet, so we freeze them.
                p.requires_grad_(False)
            else:
                # Remaining parameters are shared and stay trainable as usual.
                shared_param_names.append(name)
                p.requires_grad_(True)

        self.total_generated_params = flat_offset
        if self.total_generated_params == 0:
            raise ValueError(
                f"Zero parameters defined by the hypernetwork. "
                f"Check target_modules: {self.target_modules}."
            )
        
        # Cache a dict of the shared parameters for quick access in forward().
        self._shared_param_dict = {
            name: p
            for name, p in self.base_ae.named_parameters()
            if name in shared_param_names
        }

        # Hypernetwork input dimension is the flattened per-sample input.
        in_dims = input_sample[0].numel()
        self.hypernet = nn.Sequential(
            nn.Flatten(1),
            nn.Linear(in_dims, self.hidden_dim),
            nn.ReLU(),
            nn.Linear(self.hidden_dim, self.hidden_dim),
            nn.ReLU(),
            nn.Linear(self.hidden_dim, self.total_generated_params),
        )

        # Precompute in_dims mapping for vmap: generated params vary on dim 0, shared are None.
        # Use the same key ordering as `params` in forward: shared first, then generated
        generated_params_names = [name for (name, _, _, _) in self._param_info]

        all_param_names = list(self._shared_param_dict.keys()) + generated_params_names

        self._in_dims_params = {
            name: (0 if name in generated_params_names else None)
            for name in all_param_names
        }

        # Prebuild the vmapped single-sample call.
        self._vmapped_call = torch.func.vmap(
            self._call_single,
            in_dims=(self._in_dims_params, 0),
            out_dims=0,
            randomness="different",
        )

        self._built = True

    def _generated_params(self, x: torch.Tensor) -> Dict[str, torch.Tensor]:
        """
        Run the hypernetwork and reshape its output into per-parameter tensors.

        Args:
            x:
                Input batch of shape (B, ...) used to condition the hypernetwork.

        Returns:
            A dictionary mapping parameter names (inside `target_modules`) to
            tensors of shape (B, *original_shape), i.e. one generated parameter
            tensor per sample in the batch.
        """
        B = x.size(0)
        flat = self.hypernet(x) # type: ignore -- (B, total_generated_params)

        gen: Dict[str, torch.Tensor] = {}
        for name, shape, start, end in self._param_info:
            slice_ = flat[:, start:end]
            gen[name] = slice_.view(B, *shape)
        return gen
    
    def _call_single(self, params_i: Dict[str, torch.Tensor], x_i: torch.Tensor, **kwargs: Any):
        """
        Apply the base autoencoder to a single sample with a single param set.

        Args:
            params_i:
                Parameter dict for this particular sample.
            x_i:
                Input tensor of shape (...,) for this sample.

        Returns:
            A tuple of output tensors corresponding to the `ModelOutput` fields,
            with the leading batch dimension (size 1) removed.
        """
        # print("encoder.1.weight in _call_single:", params_i["encoder.1.weight"].shape)
        # print("encoder.1.bias in _call_single:", params_i["encoder.1.bias"].shape)
        # print("decoder.0.weight in _call_single:", params_i["decoder.0.weight"].shape)
        # print("decoder.0.bias in _call_single:", params_i["decoder.0.bias"].shape)
        out = torch.func.functional_call(
            self.base_ae,
            params_i,
            (x_i.unsqueeze(0),),
            kwargs=kwargs,
        )

        # Construct a new output of the same type, but with squeezed tensors.
        squeezed_kwargs: Dict[str, Any] = {}
        for f in self._output_fields: # type: ignore
            v = getattr(out, f.name)
            if torch.is_tensor(v):
                squeezed_kwargs[f.name] = v.squeeze(0)
            else:
                squeezed_kwargs[f.name] = v

        out_squeezed = self._output_type(**squeezed_kwargs) # type: ignore

        # Convert the structured output into a tuple of tensors (for vmap).
        tensors_tuple = tuple(
            getattr(out_squeezed, name) for name in self._output_field_names # type: ignore
        )
        return tensors_tuple

    def forward(self, x: torch.Tensor, **kwargs: Any):
        """
        Forward pass with per-sample generated parameters.

        For each sample in the batch:
          - generate a distinct set of parameters for `target_modules`
          - combine them with the shared parameters
          - call the underlying `base_ae` via `torch.func.functional_call`
        The underlying `base_ae` is evaluated in a batched, vectorized way using
        `torch.func.vmap`.

        Args:
            x:
                Input batch of shape (B, ...).
            **kwargs:
                Additional keyword arguments forwarded to `base_ae.forward`.

        Returns:
            A `ModelOutput` instance of the same type/structure as produced by
            `base_ae`, but with each field batched over the leading dimension.
        """
        # Shared parameters are the same for all samples.
        shared = self._shared_param_dict
        # Generated parameters are per-sample (B, *shape)
        generated = self._generated_params(x)

        # Combined view of all parameters (names mapped to tensors).
        params: Dict[str, torch.Tensor] = {}
        params.update(shared)
        params.update(generated)

        # Vectorized application over batch of (params, x) using prebuilt vmap.
        batched_tensors_tuple = self._vmapped_call(params, x, **kwargs) # type: ignore
        
        # Reconstruct a batched `ModelOutput` of the same type as the base AE.
        batched_kwargs = {
            name: tensor
            for name, tensor in zip(self._output_field_names, batched_tensors_tuple)  # type: ignore
        }
        batched_out = self._output_type(**batched_kwargs)  # type: ignore

        return batched_out
    