"use strict";
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const cpactions = require("../../lib");
function _assertActionMatches(test, stack, actions, provider, category, configuration) {
    const configurationStr = configuration
        ? `, configuration including ${JSON.stringify(stack.resolve(configuration), null, 2)}`
        : '';
    const actionsStr = JSON.stringify(actions.map(a => ({ owner: a.actionProperties.owner, provider: a.actionProperties.provider,
        category: a.actionProperties.category, configuration: stack.resolve(a.actionConfig.configuration),
    })), null, 2);
    test.ok(_hasAction(stack, actions, provider, category, configuration), `Expected to find an action with provider ${provider}, category ${category}${configurationStr}, but found ${actionsStr}`);
}
function _hasAction(stack, actions, provider, category, configuration) {
    for (const action of actions) {
        if (action.actionProperties.provider !== provider) {
            continue;
        }
        if (action.actionProperties.category !== category) {
            continue;
        }
        if (configuration && !action.actionConfig.configuration) {
            continue;
        }
        if (configuration) {
            for (const key of Object.keys(configuration)) {
                if (!_.isEqual(stack.resolve(action.actionConfig.configuration[key]), stack.resolve(configuration[key]))) {
                    continue;
                }
            }
        }
        return true;
    }
    return false;
}
function _assertPermissionGranted(test, stack, statements, action, resource, conditions) {
    const conditionStr = conditions
        ? ` with condition(s) ${JSON.stringify(stack.resolve(conditions))}`
        : '';
    const resolvedStatements = stack.resolve(statements.map(s => s.toStatementJson()));
    const statementsStr = JSON.stringify(resolvedStatements, null, 2);
    test.ok(_grantsPermission(stack, resolvedStatements, action, resource, conditions), `Expected to find a statement granting ${action} on ${JSON.stringify(stack.resolve(resource))}${conditionStr}, found:\n${statementsStr}`);
}
function _grantsPermission(stack, statements, action, resource, conditions) {
    for (const statement of statements.filter(s => s.Effect === 'Allow')) {
        if (!_isOrContains(stack, statement.Action, action)) {
            continue;
        }
        if (!_isOrContains(stack, statement.Resource, resource)) {
            continue;
        }
        if (conditions && !_isOrContains(stack, statement.Condition, conditions)) {
            continue;
        }
        return true;
    }
    return false;
}
function _isOrContains(stack, entity, value) {
    const resolvedValue = stack.resolve(value);
    const resolvedEntity = stack.resolve(entity);
    if (_.isEqual(resolvedEntity, resolvedValue)) {
        return true;
    }
    if (!Array.isArray(resolvedEntity)) {
        return false;
    }
    for (const tested of entity) {
        if (_.isEqual(tested, resolvedValue)) {
            return true;
        }
    }
    return false;
}
function _stackArn(stackName, scope) {
    return cdk.Stack.of(scope).formatArn({
        service: 'cloudformation',
        resource: 'stack',
        resourceName: `${stackName}/*`,
    });
}
class PipelineDouble extends cdk.Resource {
    constructor(scope, id, { pipelineName, role }) {
        super(scope, id);
        this.pipelineName = pipelineName || 'TestPipeline';
        this.pipelineArn = cdk.Stack.of(this).formatArn({ service: 'codepipeline', resource: 'pipeline', resourceName: this.pipelineName });
        this.role = role;
        this.artifactBucket = new BucketDouble(scope, 'BucketDouble');
    }
    onEvent(_id, _options) {
        throw new Error('Method not implemented.');
    }
    onStateChange(_id, _options) {
        throw new Error('Method not implemented.');
    }
}
class FullAction {
    constructor(actionProperties, actionConfig) {
        this.actionProperties = actionProperties;
        this.actionConfig = actionConfig;
        // empty
    }
}
class StageDouble {
    constructor({ name, pipeline, actions }) {
        this.actions = [];
        this.stageName = name || 'TestStage';
        this.pipeline = pipeline;
        const stageParent = new cdk.Construct(pipeline, this.stageName);
        const fullActions = new Array();
        for (const action of actions) {
            const actionParent = new cdk.Construct(stageParent, action.actionProperties.actionName);
            fullActions.push(new FullAction(action.actionProperties, action.bind(actionParent, this, {
                role: pipeline.role,
                bucket: pipeline.artifactBucket,
            })));
        }
        this.fullActions = fullActions;
    }
    get node() {
        throw new Error('StageDouble is not a real construct');
    }
    addAction(_action) {
        throw new Error('addAction() is not supported on StageDouble');
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not supported on StageDouble');
    }
}
class RoleDouble extends iam.Role {
    constructor(scope, id, props = { assumedBy: new iam.ServicePrincipal('test') }) {
        super(scope, id, props);
        this.statements = new Array();
    }
    addToPolicy(statement) {
        super.addToPolicy(statement);
        this.statements.push(statement);
        return true;
    }
}
class BucketDouble extends s3.Bucket {
    grantRead(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantReadWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
}
module.exports = nodeunit.testCase({
    'CreateReplaceChangeSet': {
        'works'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Stack');
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            const action = new cpactions.CloudFormationCreateReplaceChangeSetAction({
                actionName: 'Action',
                changeSetName: 'MyChangeSet',
                stackName: 'MyStack',
                templatePath: artifact.atPath('path/to/file'),
                adminPermissions: false,
            });
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [action],
            });
            cdk.ConstructNode.prepare(stack.node);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
            const stackArn = _stackArn('MyStack', stack);
            const changeSetCondition = { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } };
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStacks', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteChangeSet', stackArn, changeSetCondition);
            // TODO: revert "as any" once we move all actions into a single package.
            test.deepEqual(stage.fullActions[0].actionProperties.inputs, [artifact], 'The input was correctly registered');
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_CREATE_REPLACE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet',
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/file'),
                    }),
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/other/file'),
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::GetAtt': ['PipelineTestStageActionARole9283FBE3', 'Arn'] },
                        { 'Fn::GetAtt': ['PipelineTestStageActionBRoleCABC8FA5', 'Arn'] },
                    ],
                },
                {
                    Action: [
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] },
                    ],
                },
            ]);
            test.done();
        },
    },
    'ExecuteChangeSet': {
        'works'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action',
                        changeSetName: 'MyChangeSet',
                        stackName: 'MyStack',
                    }),
                ],
            });
            const stackArn = _stackArn('MyStack', stack);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:ExecuteChangeSet', stackArn, { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } });
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_EXECUTE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet',
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: [
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                        'cloudformation:ExecuteChangeSet',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] },
                    ],
                },
            ]);
            test.done();
        },
    },
    'the CreateUpdateStack Action sets the DescribeStack*, Create/Update/DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'Action',
            templatePath: new codepipeline.Artifact('TestArtifact').atPath('some/file'),
            stackName: 'MyStack',
            adminPermissions: false,
            replaceOnFailure: true,
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:UpdateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
    'the DeleteStack Action sets the DescribeStack*, DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationDeleteStackAction({
            actionName: 'Action',
            adminPermissions: false,
            stackName: 'MyStack',
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5waXBlbGluZS1hY3Rpb25zLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5waXBlbGluZS1hY3Rpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSwwREFBMEQ7QUFFMUQsd0NBQXdDO0FBQ3hDLHNDQUFzQztBQUN0QyxxQ0FBcUM7QUFDckMsNEJBQTRCO0FBQzVCLHFDQUFxQztBQUNyQyx1Q0FBdUM7QUF1T3ZDLFNBQVMsb0JBQW9CLENBQzNCLElBQW1CLEVBQ25CLEtBQWdCLEVBQ2hCLE9BQXFCLEVBQ3JCLFFBQWdCLEVBQ2hCLFFBQWdCLEVBQ2hCLGFBQXNDO0lBQ3RDLE1BQU0sZ0JBQWdCLEdBQUcsYUFBYTtRQUNwQyxDQUFDLENBQUMsNkJBQTZCLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUU7UUFDdEYsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNQLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUNoRCxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRO1FBQ3ZFLFFBQVEsRUFBRSxDQUFDLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDO0tBQ2xHLENBQUMsQ0FDSCxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNaLElBQUksQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxhQUFhLENBQUMsRUFDbkUsNENBQTRDLFFBQVEsY0FBYyxRQUFRLEdBQUcsZ0JBQWdCLGVBQWUsVUFBVSxFQUFFLENBQUMsQ0FBQztBQUM5SCxDQUFDO0FBRUQsU0FBUyxVQUFVLENBQ2pCLEtBQWdCLEVBQUUsT0FBcUIsRUFBRSxRQUFnQixFQUFFLFFBQWdCLEVBQzNFLGFBQXFDO0lBQ3JDLEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1FBQzVCLElBQUksTUFBTSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsS0FBSyxRQUFRLEVBQUU7WUFBRSxTQUFTO1NBQUU7UUFDaEUsSUFBSSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxLQUFLLFFBQVEsRUFBRTtZQUFFLFNBQVM7U0FBRTtRQUNoRSxJQUFJLGFBQWEsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsYUFBYSxFQUFFO1lBQUUsU0FBUztTQUFFO1FBQ3RFLElBQUksYUFBYSxFQUFFO1lBQ2pCLEtBQUssTUFBTSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtnQkFDNUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRTtvQkFDeEcsU0FBUztpQkFDVjthQUNGO1NBQ0Y7UUFDRCxPQUFPLElBQUksQ0FBQztLQUNiO0lBQ0QsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBRUQsU0FBUyx3QkFBd0IsQ0FDL0IsSUFBbUIsRUFDbkIsS0FBZ0IsRUFDaEIsVUFBaUMsRUFDakMsTUFBYyxFQUNkLFFBQWdCLEVBQ2hCLFVBQWdCO0lBQ2hCLE1BQU0sWUFBWSxHQUFHLFVBQVU7UUFDN0IsQ0FBQyxDQUFDLHNCQUFzQixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsRUFBRTtRQUNuRSxDQUFDLENBQUMsRUFBRSxDQUFDO0lBQ1AsTUFBTSxrQkFBa0IsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25GLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsa0JBQWtCLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ2xFLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsVUFBVSxDQUFDLEVBQ2hGLHlDQUF5QyxNQUFNLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLEdBQUcsWUFBWSxhQUFhLGFBQWEsRUFBRSxDQUFDLENBQUM7QUFDOUksQ0FBQztBQUVELFNBQVMsaUJBQWlCLENBQUMsS0FBZ0IsRUFBRSxVQUFpQyxFQUFFLE1BQWMsRUFBRSxRQUFnQixFQUFFLFVBQWdCO0lBQ2hJLEtBQUssTUFBTSxTQUFTLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLEtBQUssT0FBTyxDQUFDLEVBQUU7UUFDcEUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsRUFBRTtZQUFFLFNBQVM7U0FBRTtRQUNsRSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxFQUFFO1lBQUUsU0FBUztTQUFFO1FBQ3RFLElBQUksVUFBVSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsU0FBUyxFQUFFLFVBQVUsQ0FBQyxFQUFFO1lBQUUsU0FBUztTQUFFO1FBQ3ZGLE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFDRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FBQyxLQUFnQixFQUFFLE1BQXlCLEVBQUUsS0FBYTtJQUMvRSxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDN0MsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxhQUFhLENBQUMsRUFBRTtRQUFFLE9BQU8sSUFBSSxDQUFDO0tBQUU7SUFDOUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEVBQUU7UUFBRSxPQUFPLEtBQUssQ0FBQztLQUFFO0lBQ3JELEtBQUssTUFBTSxNQUFNLElBQUksTUFBTSxFQUFFO1FBQzNCLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLEVBQUU7WUFBRSxPQUFPLElBQUksQ0FBQztTQUFFO0tBQ3ZEO0lBQ0QsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBRUQsU0FBUyxTQUFTLENBQUMsU0FBaUIsRUFBRSxLQUFxQjtJQUN6RCxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUNuQyxPQUFPLEVBQUUsZ0JBQWdCO1FBQ3pCLFFBQVEsRUFBRSxPQUFPO1FBQ2pCLFlBQVksRUFBRSxHQUFHLFNBQVMsSUFBSTtLQUMvQixDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsTUFBTSxjQUFlLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFNdkMsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQTZDO1FBQzdHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLElBQUksY0FBYyxDQUFDO1FBQ25ELElBQUksQ0FBQyxXQUFXLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUNwSSxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRU0sT0FBTyxDQUFDLEdBQVcsRUFBRSxRQUErQjtRQUN6RCxNQUFNLElBQUksS0FBSyxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUNNLGFBQWEsQ0FBQyxHQUFXLEVBQUUsUUFBK0I7UUFDL0QsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7Q0FDRjtBQUVELE1BQU0sVUFBVTtJQUNkLFlBQ1csZ0JBQStDLEVBQy9DLFlBQXVDO1FBRHZDLHFCQUFnQixHQUFoQixnQkFBZ0IsQ0FBK0I7UUFDL0MsaUJBQVksR0FBWixZQUFZLENBQTJCO1FBQ2hELFFBQVE7SUFDVixDQUFDO0NBQ0Y7QUFFRCxNQUFNLFdBQVc7SUFVZixZQUFZLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQWdGO1FBUHJHLFlBQU8sR0FBMkIsRUFBRSxDQUFDO1FBUW5ELElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxJQUFJLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUV6QixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoRSxNQUFNLFdBQVcsR0FBRyxJQUFJLEtBQUssRUFBYyxDQUFDO1FBQzVDLEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzVCLE1BQU0sWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3hGLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLElBQUksRUFBRTtnQkFDdkYsSUFBSSxFQUFFLFFBQVEsQ0FBQyxJQUFJO2dCQUNuQixNQUFNLEVBQUUsUUFBUSxDQUFDLGNBQWM7YUFDaEMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNOO1FBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUM7SUFDakMsQ0FBQztJQWxCRCxJQUFXLElBQUk7UUFDYixNQUFNLElBQUksS0FBSyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQWtCTSxTQUFTLENBQUMsT0FBNkI7UUFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFTSxhQUFhLENBQUMsS0FBYSxFQUFFLE9BQTRCLEVBQUUsUUFBMkI7UUFFM0YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO0lBQ3JFLENBQUM7Q0FDRjtBQUVELE1BQU0sVUFBVyxTQUFRLEdBQUcsQ0FBQyxJQUFJO0lBRy9CLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsUUFBdUIsRUFBRSxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDbEgsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFIVixlQUFVLEdBQUcsSUFBSSxLQUFLLEVBQXVCLENBQUM7SUFJOUQsQ0FBQztJQUVNLFdBQVcsQ0FBQyxTQUE4QjtRQUMvQyxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzdCLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGO0FBRUQsTUFBTSxZQUFhLFNBQVEsRUFBRSxDQUFDLE1BQU07SUFDM0IsU0FBUyxDQUFDLFFBQXdCLEVBQUUscUJBQTBCLEdBQUc7UUFDdEUsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVNLFVBQVUsQ0FBQyxRQUF3QixFQUFFLHFCQUEwQixHQUFHO1FBQ3ZFLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFTSxjQUFjLENBQUMsUUFBd0IsRUFBRSxxQkFBMEIsR0FBRztRQUMzRSxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN0QyxDQUFDO0NBQ0Y7QUFwWkQsaUJBQVMsUUFBUSxDQUFDLFFBQVEsQ0FBQztJQUN6Qix3QkFBd0IsRUFBRTtRQUN4QixPQUFPLENBQUMsSUFBbUI7WUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUMxQyxNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDM0QsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQzNELE1BQU0sTUFBTSxHQUFHLElBQUksU0FBUyxDQUFDLDBDQUEwQyxDQUFDO2dCQUN0RSxVQUFVLEVBQUUsUUFBUTtnQkFDcEIsYUFBYSxFQUFFLGFBQWE7Z0JBQzVCLFNBQVMsRUFBRSxTQUFTO2dCQUNwQixZQUFZLEVBQUUsUUFBUSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUM7Z0JBQzdDLGdCQUFnQixFQUFFLEtBQUs7YUFDeEIsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFXLENBQUM7Z0JBQzVCLFFBQVEsRUFBRSxJQUFJLGNBQWMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxDQUFDO2dCQUN2RSxPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUM7YUFDbEIsQ0FBQyxDQUFDO1lBRUgsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBRXRDLHdCQUF3QixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVUsRUFBRSxjQUFjLEVBQUUsTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUU5RyxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQzdDLE1BQU0sa0JBQWtCLEdBQUcsRUFBRSxvQkFBb0IsRUFBRSxFQUFFLDhCQUE4QixFQUFFLGFBQWEsRUFBRSxFQUFFLENBQUM7WUFDdkcsd0JBQXdCLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxZQUFZLENBQUMsVUFBVSxFQUFFLCtCQUErQixFQUFFLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1lBQzlILHdCQUF3QixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVUsRUFBRSxrQ0FBa0MsRUFBRSxRQUFRLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztZQUNqSSx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLFlBQVksQ0FBQyxVQUFVLEVBQUUsZ0NBQWdDLEVBQUUsUUFBUSxFQUFFLGtCQUFrQixDQUFDLENBQUM7WUFDL0gsd0JBQXdCLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxZQUFZLENBQUMsVUFBVSxFQUFFLGdDQUFnQyxFQUFFLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1lBRS9ILHdFQUF3RTtZQUN4RSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQ3JFLG9DQUFvQyxDQUFDLENBQUM7WUFFeEMsb0JBQW9CLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLGdCQUFnQixFQUFFLFFBQVEsRUFBRTtnQkFDL0UsVUFBVSxFQUFFLDJCQUEyQjtnQkFDdkMsU0FBUyxFQUFFLFNBQVM7Z0JBQ3BCLGFBQWEsRUFBRSxhQUFhO2FBQzdCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx1RUFBdUUsQ0FBQyxJQUFtQjtZQUN6RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDM0QsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQzNELElBQUksV0FBVyxDQUFDO2dCQUNkLFFBQVEsRUFBRSxJQUFJLGNBQWMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxDQUFDO2dCQUN2RSxPQUFPLEVBQUU7b0JBQ1AsSUFBSSxTQUFTLENBQUMsMENBQTBDLENBQUM7d0JBQ3ZELFVBQVUsRUFBRSxTQUFTO3dCQUNyQixhQUFhLEVBQUUsYUFBYTt3QkFDNUIsU0FBUyxFQUFFLFFBQVE7d0JBQ25CLGdCQUFnQixFQUFFLEtBQUs7d0JBQ3ZCLFlBQVksRUFBRSxRQUFRLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQztxQkFDOUMsQ0FBQztvQkFDRixJQUFJLFNBQVMsQ0FBQywwQ0FBMEMsQ0FBQzt3QkFDdkQsVUFBVSxFQUFFLFNBQVM7d0JBQ3JCLGFBQWEsRUFBRSxhQUFhO3dCQUM1QixTQUFTLEVBQUUsUUFBUTt3QkFDbkIsZ0JBQWdCLEVBQUUsS0FBSzt3QkFDdkIsWUFBWSxFQUFFLFFBQVEsQ0FBQyxNQUFNLENBQUMsb0JBQW9CLENBQUM7cUJBQ3BELENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsU0FBUyxDQUNaLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxFQUNwRTtnQkFDRTtvQkFDRSxNQUFNLEVBQUUsY0FBYztvQkFDdEIsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLEVBQUUsWUFBWSxFQUFFLENBQUUsc0NBQXNDLEVBQUUsS0FBSyxDQUFFLEVBQUU7d0JBQ25FLEVBQUUsWUFBWSxFQUFFLENBQUUsc0NBQXNDLEVBQUUsS0FBSyxDQUFFLEVBQUU7cUJBQ3BFO2lCQUNGO2dCQUNEO29CQUNFLE1BQU0sRUFBRTt3QkFDTixnQ0FBZ0M7d0JBQ2hDLGdDQUFnQzt3QkFDaEMsa0NBQWtDO3dCQUNsQywrQkFBK0I7cUJBQ2hDO29CQUNELFNBQVMsRUFBRSxFQUFFLG9CQUFvQixFQUFFLEVBQUUsOEJBQThCLEVBQUUsYUFBYSxFQUFFLEVBQUU7b0JBQ3RGLE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUiwyQ0FBMkM7d0JBQzNDLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsaUJBQWlCLENBQUUsQ0FBRSxFQUFFO3dCQUMxSiwyQ0FBMkM7d0JBQzNDLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsaUJBQWlCLENBQUUsQ0FBRSxFQUFFO3FCQUMzSjtpQkFDRjthQUNGLENBQ0YsQ0FBQztZQUVGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELGtCQUFrQixFQUFFO1FBQ2xCLE9BQU8sQ0FBQyxJQUFtQjtZQUN6QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDM0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFXLENBQUM7Z0JBQzVCLFFBQVEsRUFBRSxJQUFJLGNBQWMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxDQUFDO2dCQUN2RSxPQUFPLEVBQUU7b0JBQ1AsSUFBSSxTQUFTLENBQUMsb0NBQW9DLENBQUM7d0JBQ2pELFVBQVUsRUFBRSxRQUFRO3dCQUNwQixhQUFhLEVBQUUsYUFBYTt3QkFDNUIsU0FBUyxFQUFFLFNBQVM7cUJBQ3JCLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQzdDLHdCQUF3QixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVUsRUFBRSxpQ0FBaUMsRUFBRSxRQUFRLEVBQ3hHLEVBQUUsb0JBQW9CLEVBQUUsRUFBRSw4QkFBOEIsRUFBRSxhQUFhLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFFL0Usb0JBQW9CLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLGdCQUFnQixFQUFFLFFBQVEsRUFBRTtnQkFDL0UsVUFBVSxFQUFFLG9CQUFvQjtnQkFDaEMsU0FBUyxFQUFFLFNBQVM7Z0JBQ3BCLGFBQWEsRUFBRSxhQUFhO2FBQzdCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx1RUFBdUUsQ0FBQyxJQUFtQjtZQUN6RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDM0QsSUFBSSxXQUFXLENBQUM7Z0JBQ2QsUUFBUSxFQUFFLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLENBQUM7Z0JBQ3ZFLE9BQU8sRUFBRTtvQkFDUCxJQUFJLFNBQVMsQ0FBQyxvQ0FBb0MsQ0FBQzt3QkFDakQsVUFBVSxFQUFFLFNBQVM7d0JBQ3JCLGFBQWEsRUFBRSxhQUFhO3dCQUM1QixTQUFTLEVBQUUsUUFBUTtxQkFDcEIsQ0FBQztvQkFDRixJQUFJLFNBQVMsQ0FBQyxvQ0FBb0MsQ0FBQzt3QkFDakQsVUFBVSxFQUFFLFNBQVM7d0JBQ3JCLGFBQWEsRUFBRSxhQUFhO3dCQUM1QixTQUFTLEVBQUUsUUFBUTtxQkFDcEIsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxTQUFTLENBQ1osS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLEVBQ3BFO2dCQUNFO29CQUNFLE1BQU0sRUFBRTt3QkFDTixrQ0FBa0M7d0JBQ2xDLCtCQUErQjt3QkFDL0IsaUNBQWlDO3FCQUNsQztvQkFDRCxTQUFTLEVBQUUsRUFBRSxvQkFBb0IsRUFBRSxFQUFFLDhCQUE4QixFQUFFLGFBQWEsRUFBRSxFQUFFO29CQUN0RixNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUU7d0JBQ1IsMkNBQTJDO3dCQUMzQyxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGlCQUFpQixDQUFFLENBQUUsRUFBRTt3QkFDMUosMkNBQTJDO3dCQUMzQyxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLGlCQUFpQixDQUFFLENBQUUsRUFBRTtxQkFDM0o7aUJBQ0Y7YUFDRixDQUNGLENBQUM7WUFFRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCx3R0FBd0csQ0FBQyxJQUFtQjtRQUMxSCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDM0QsTUFBTSxNQUFNLEdBQUcsSUFBSSxTQUFTLENBQUMscUNBQXFDLENBQUM7WUFDakUsVUFBVSxFQUFFLFFBQVE7WUFDcEIsWUFBWSxFQUFFLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDO1lBQzNFLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDLENBQUM7UUFDSCxJQUFJLFdBQVcsQ0FBQztZQUNkLFFBQVEsRUFBRSxJQUFJLGNBQWMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxDQUFDO1lBQ3ZFLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztTQUNsQixDQUFDLENBQUM7UUFDSCxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRTdDLHdCQUF3QixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVUsRUFBRSwrQkFBK0IsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMxRyx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLFlBQVksQ0FBQyxVQUFVLEVBQUUsNEJBQTRCLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdkcsd0JBQXdCLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxZQUFZLENBQUMsVUFBVSxFQUFFLDRCQUE0QixFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3ZHLHdCQUF3QixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVUsRUFBRSw0QkFBNEIsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUV2Ryx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLFlBQVksQ0FBQyxVQUFVLEVBQUUsY0FBYyxFQUFFLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFOUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG9GQUFvRixDQUFDLElBQW1CO1FBQ3RHLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sWUFBWSxHQUFHLElBQUksVUFBVSxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQztRQUMzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLFNBQVMsQ0FBQywrQkFBK0IsQ0FBQztZQUMzRCxVQUFVLEVBQUUsUUFBUTtZQUNwQixnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLFNBQVMsRUFBRSxTQUFTO1NBQ3JCLENBQUMsQ0FBQztRQUNILElBQUksV0FBVyxDQUFDO1lBQ2QsUUFBUSxFQUFFLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLENBQUM7WUFDdkUsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDO1NBQ2xCLENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFN0Msd0JBQXdCLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxZQUFZLENBQUMsVUFBVSxFQUFFLCtCQUErQixFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzFHLHdCQUF3QixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVUsRUFBRSw0QkFBNEIsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUV2Ryx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLFlBQVksQ0FBQyxVQUFVLEVBQUUsY0FBYyxFQUFFLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFOUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBfIGZyb20gJ2xvZGFzaCc7XG5pbXBvcnQgKiBhcyBub2RldW5pdCBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBjcGFjdGlvbnMgZnJvbSAnLi4vLi4vbGliJztcblxuZXhwb3J0ID0gbm9kZXVuaXQudGVzdENhc2Uoe1xuICAnQ3JlYXRlUmVwbGFjZUNoYW5nZVNldCc6IHtcbiAgICAnd29ya3MnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICAgIGNvbnN0IGFwcCA9IG5ldyBjZGsuQXBwKCk7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnU3RhY2snKTtcbiAgICAgIGNvbnN0IHBpcGVsaW5lUm9sZSA9IG5ldyBSb2xlRG91YmxlKHN0YWNrLCAnUGlwZWxpbmVSb2xlJyk7XG4gICAgICBjb25zdCBhcnRpZmFjdCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoJ1Rlc3RBcnRpZmFjdCcpO1xuICAgICAgY29uc3QgYWN0aW9uID0gbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgICBhY3Rpb25OYW1lOiAnQWN0aW9uJyxcbiAgICAgICAgY2hhbmdlU2V0TmFtZTogJ015Q2hhbmdlU2V0JyxcbiAgICAgICAgc3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICAgIHRlbXBsYXRlUGF0aDogYXJ0aWZhY3QuYXRQYXRoKCdwYXRoL3RvL2ZpbGUnKSxcbiAgICAgICAgYWRtaW5QZXJtaXNzaW9uczogZmFsc2UsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHN0YWdlID0gbmV3IFN0YWdlRG91YmxlKHtcbiAgICAgICAgcGlwZWxpbmU6IG5ldyBQaXBlbGluZURvdWJsZShzdGFjaywgJ1BpcGVsaW5lJywgeyByb2xlOiBwaXBlbGluZVJvbGUgfSksXG4gICAgICAgIGFjdGlvbnM6IFthY3Rpb25dLFxuICAgICAgfSk7XG5cbiAgICAgIGNkay5Db25zdHJ1Y3ROb2RlLnByZXBhcmUoc3RhY2subm9kZSk7XG5cbiAgICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdpYW06UGFzc1JvbGUnLCBhY3Rpb24uZGVwbG95bWVudFJvbGUucm9sZUFybik7XG5cbiAgICAgIGNvbnN0IHN0YWNrQXJuID0gX3N0YWNrQXJuKCdNeVN0YWNrJywgc3RhY2spO1xuICAgICAgY29uc3QgY2hhbmdlU2V0Q29uZGl0aW9uID0geyBTdHJpbmdFcXVhbHNJZkV4aXN0czogeyAnY2xvdWRmb3JtYXRpb246Q2hhbmdlU2V0TmFtZSc6ICdNeUNoYW5nZVNldCcgfSB9O1xuICAgICAgX2Fzc2VydFBlcm1pc3Npb25HcmFudGVkKHRlc3QsIHN0YWNrLCBwaXBlbGluZVJvbGUuc3RhdGVtZW50cywgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tzJywgc3RhY2tBcm4sIGNoYW5nZVNldENvbmRpdGlvbik7XG4gICAgICBfYXNzZXJ0UGVybWlzc2lvbkdyYW50ZWQodGVzdCwgc3RhY2ssIHBpcGVsaW5lUm9sZS5zdGF0ZW1lbnRzLCAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVDaGFuZ2VTZXQnLCBzdGFja0FybiwgY2hhbmdlU2V0Q29uZGl0aW9uKTtcbiAgICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdjbG91ZGZvcm1hdGlvbjpDcmVhdGVDaGFuZ2VTZXQnLCBzdGFja0FybiwgY2hhbmdlU2V0Q29uZGl0aW9uKTtcbiAgICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdjbG91ZGZvcm1hdGlvbjpEZWxldGVDaGFuZ2VTZXQnLCBzdGFja0FybiwgY2hhbmdlU2V0Q29uZGl0aW9uKTtcblxuICAgICAgLy8gVE9ETzogcmV2ZXJ0IFwiYXMgYW55XCIgb25jZSB3ZSBtb3ZlIGFsbCBhY3Rpb25zIGludG8gYSBzaW5nbGUgcGFja2FnZS5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWdlLmZ1bGxBY3Rpb25zWzBdLmFjdGlvblByb3BlcnRpZXMuaW5wdXRzLCBbYXJ0aWZhY3RdLFxuICAgICAgICAnVGhlIGlucHV0IHdhcyBjb3JyZWN0bHkgcmVnaXN0ZXJlZCcpO1xuXG4gICAgICBfYXNzZXJ0QWN0aW9uTWF0Y2hlcyh0ZXN0LCBzdGFjaywgc3RhZ2UuZnVsbEFjdGlvbnMsICdDbG91ZEZvcm1hdGlvbicsICdEZXBsb3knLCB7XG4gICAgICAgIEFjdGlvbk1vZGU6ICdDSEFOR0VfU0VUX0NSRUFURV9SRVBMQUNFJyxcbiAgICAgICAgU3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICAgIENoYW5nZVNldE5hbWU6ICdNeUNoYW5nZVNldCcsXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd1c2VzIGEgc2luZ2xlIHBlcm1pc3Npb24gc3RhdGVtZW50IGlmIHRoZSBzYW1lIENoYW5nZVNldCBuYW1lIGlzIHVzZWQnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgcGlwZWxpbmVSb2xlID0gbmV3IFJvbGVEb3VibGUoc3RhY2ssICdQaXBlbGluZVJvbGUnKTtcbiAgICAgIGNvbnN0IGFydGlmYWN0ID0gbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdCgnVGVzdEFydGlmYWN0Jyk7XG4gICAgICBuZXcgU3RhZ2VEb3VibGUoe1xuICAgICAgICBwaXBlbGluZTogbmV3IFBpcGVsaW5lRG91YmxlKHN0YWNrLCAnUGlwZWxpbmUnLCB7IHJvbGU6IHBpcGVsaW5lUm9sZSB9KSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgIG5ldyBjcGFjdGlvbnMuQ2xvdWRGb3JtYXRpb25DcmVhdGVSZXBsYWNlQ2hhbmdlU2V0QWN0aW9uKHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdBY3Rpb25BJyxcbiAgICAgICAgICAgIGNoYW5nZVNldE5hbWU6ICdNeUNoYW5nZVNldCcsXG4gICAgICAgICAgICBzdGFja05hbWU6ICdTdGFja0EnLFxuICAgICAgICAgICAgYWRtaW5QZXJtaXNzaW9uczogZmFsc2UsXG4gICAgICAgICAgICB0ZW1wbGF0ZVBhdGg6IGFydGlmYWN0LmF0UGF0aCgncGF0aC90by9maWxlJyksXG4gICAgICAgICAgfSksXG4gICAgICAgICAgbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0FjdGlvbkInLFxuICAgICAgICAgICAgY2hhbmdlU2V0TmFtZTogJ015Q2hhbmdlU2V0JyxcbiAgICAgICAgICAgIHN0YWNrTmFtZTogJ1N0YWNrQicsXG4gICAgICAgICAgICBhZG1pblBlcm1pc3Npb25zOiBmYWxzZSxcbiAgICAgICAgICAgIHRlbXBsYXRlUGF0aDogYXJ0aWZhY3QuYXRQYXRoKCdwYXRoL3RvL290aGVyL2ZpbGUnKSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChcbiAgICAgICAgc3RhY2sucmVzb2x2ZShwaXBlbGluZVJvbGUuc3RhdGVtZW50cy5tYXAocyA9PiBzLnRvU3RhdGVtZW50SnNvbigpKSksXG4gICAgICAgIFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdpYW06UGFzc1JvbGUnLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IFtcbiAgICAgICAgICAgICAgeyAnRm46OkdldEF0dCc6IFsgJ1BpcGVsaW5lVGVzdFN0YWdlQWN0aW9uQVJvbGU5MjgzRkJFMycsICdBcm4nIF0gfSxcbiAgICAgICAgICAgICAgeyAnRm46OkdldEF0dCc6IFsgJ1BpcGVsaW5lVGVzdFN0YWdlQWN0aW9uQlJvbGVDQUJDOEZBNScsICdBcm4nIF0gfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkNyZWF0ZUNoYW5nZVNldCcsXG4gICAgICAgICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZWxldGVDaGFuZ2VTZXQnLFxuICAgICAgICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVDaGFuZ2VTZXQnLFxuICAgICAgICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja3MnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIENvbmRpdGlvbjogeyBTdHJpbmdFcXVhbHNJZkV4aXN0czogeyAnY2xvdWRmb3JtYXRpb246Q2hhbmdlU2V0TmFtZSc6ICdNeUNoYW5nZVNldCcgfSB9LFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IFtcbiAgICAgICAgICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgICAgICAgICB7ICdGbjo6Sm9pbic6IFsnJywgWydhcm46JywgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSwgJzpjbG91ZGZvcm1hdGlvbjonLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnOicsIHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sICc6c3RhY2svU3RhY2tBLyonIF0gXSB9LFxuICAgICAgICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgICAgICAgIHsgJ0ZuOjpKb2luJzogWycnLCBbJ2FybjonLCB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LCAnOmNsb3VkZm9ybWF0aW9uOicsIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sICc6JywgeyBSZWY6ICdBV1M6OkFjY291bnRJZCcgfSwgJzpzdGFjay9TdGFja0IvKicgXSBdIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICApO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdFeGVjdXRlQ2hhbmdlU2V0Jzoge1xuICAgICd3b3JrcycodGVzdDogbm9kZXVuaXQuVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBwaXBlbGluZVJvbGUgPSBuZXcgUm9sZURvdWJsZShzdGFjaywgJ1BpcGVsaW5lUm9sZScpO1xuICAgICAgY29uc3Qgc3RhZ2UgPSBuZXcgU3RhZ2VEb3VibGUoe1xuICAgICAgICBwaXBlbGluZTogbmV3IFBpcGVsaW5lRG91YmxlKHN0YWNrLCAnUGlwZWxpbmUnLCB7IHJvbGU6IHBpcGVsaW5lUm9sZSB9KSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgIG5ldyBjcGFjdGlvbnMuQ2xvdWRGb3JtYXRpb25FeGVjdXRlQ2hhbmdlU2V0QWN0aW9uKHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdBY3Rpb24nLFxuICAgICAgICAgICAgY2hhbmdlU2V0TmFtZTogJ015Q2hhbmdlU2V0JyxcbiAgICAgICAgICAgIHN0YWNrTmFtZTogJ015U3RhY2snLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IHN0YWNrQXJuID0gX3N0YWNrQXJuKCdNeVN0YWNrJywgc3RhY2spO1xuICAgICAgX2Fzc2VydFBlcm1pc3Npb25HcmFudGVkKHRlc3QsIHN0YWNrLCBwaXBlbGluZVJvbGUuc3RhdGVtZW50cywgJ2Nsb3VkZm9ybWF0aW9uOkV4ZWN1dGVDaGFuZ2VTZXQnLCBzdGFja0FybixcbiAgICAgICAgeyBTdHJpbmdFcXVhbHNJZkV4aXN0czogeyAnY2xvdWRmb3JtYXRpb246Q2hhbmdlU2V0TmFtZSc6ICdNeUNoYW5nZVNldCcgfSB9KTtcblxuICAgICAgX2Fzc2VydEFjdGlvbk1hdGNoZXModGVzdCwgc3RhY2ssIHN0YWdlLmZ1bGxBY3Rpb25zLCAnQ2xvdWRGb3JtYXRpb24nLCAnRGVwbG95Jywge1xuICAgICAgICBBY3Rpb25Nb2RlOiAnQ0hBTkdFX1NFVF9FWEVDVVRFJyxcbiAgICAgICAgU3RhY2tOYW1lOiAnTXlTdGFjaycsXG4gICAgICAgIENoYW5nZVNldE5hbWU6ICdNeUNoYW5nZVNldCcsXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd1c2VzIGEgc2luZ2xlIHBlcm1pc3Npb24gc3RhdGVtZW50IGlmIHRoZSBzYW1lIENoYW5nZVNldCBuYW1lIGlzIHVzZWQnKHRlc3Q6IG5vZGV1bml0LlRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgcGlwZWxpbmVSb2xlID0gbmV3IFJvbGVEb3VibGUoc3RhY2ssICdQaXBlbGluZVJvbGUnKTtcbiAgICAgIG5ldyBTdGFnZURvdWJsZSh7XG4gICAgICAgIHBpcGVsaW5lOiBuZXcgUGlwZWxpbmVEb3VibGUoc3RhY2ssICdQaXBlbGluZScsIHsgcm9sZTogcGlwZWxpbmVSb2xlIH0pLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0FjdGlvbkEnLFxuICAgICAgICAgICAgY2hhbmdlU2V0TmFtZTogJ015Q2hhbmdlU2V0JyxcbiAgICAgICAgICAgIHN0YWNrTmFtZTogJ1N0YWNrQScsXG4gICAgICAgICAgfSksXG4gICAgICAgICAgbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0FjdGlvbkInLFxuICAgICAgICAgICAgY2hhbmdlU2V0TmFtZTogJ015Q2hhbmdlU2V0JyxcbiAgICAgICAgICAgIHN0YWNrTmFtZTogJ1N0YWNrQicsXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kZWVwRXF1YWwoXG4gICAgICAgIHN0YWNrLnJlc29sdmUocGlwZWxpbmVSb2xlLnN0YXRlbWVudHMubWFwKHMgPT4gcy50b1N0YXRlbWVudEpzb24oKSkpLFxuICAgICAgICBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZXNjcmliZUNoYW5nZVNldCcsXG4gICAgICAgICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrcycsXG4gICAgICAgICAgICAgICdjbG91ZGZvcm1hdGlvbjpFeGVjdXRlQ2hhbmdlU2V0JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBDb25kaXRpb246IHsgU3RyaW5nRXF1YWxzSWZFeGlzdHM6IHsgJ2Nsb3VkZm9ybWF0aW9uOkNoYW5nZVNldE5hbWUnOiAnTXlDaGFuZ2VTZXQnIH0gfSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiBbXG4gICAgICAgICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgICAgICAgeyAnRm46OkpvaW4nOiBbJycsIFsnYXJuOicsIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sICc6Y2xvdWRmb3JtYXRpb246JywgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSwgJzonLCB7IFJlZjogJ0FXUzo6QWNjb3VudElkJyB9LCAnOnN0YWNrL1N0YWNrQS8qJyBdIF0gfSxcbiAgICAgICAgICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgICAgICAgICB7ICdGbjo6Sm9pbic6IFsnJywgWydhcm46JywgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSwgJzpjbG91ZGZvcm1hdGlvbjonLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnOicsIHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sICc6c3RhY2svU3RhY2tCLyonIF0gXSB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICAndGhlIENyZWF0ZVVwZGF0ZVN0YWNrIEFjdGlvbiBzZXRzIHRoZSBEZXNjcmliZVN0YWNrKiwgQ3JlYXRlL1VwZGF0ZS9EZWxldGVTdGFjayAmIFBhc3NSb2xlIHBlcm1pc3Npb25zJyh0ZXN0OiBub2RldW5pdC5UZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgcGlwZWxpbmVSb2xlID0gbmV3IFJvbGVEb3VibGUoc3RhY2ssICdQaXBlbGluZVJvbGUnKTtcbiAgICBjb25zdCBhY3Rpb24gPSBuZXcgY3BhY3Rpb25zLkNsb3VkRm9ybWF0aW9uQ3JlYXRlVXBkYXRlU3RhY2tBY3Rpb24oe1xuICAgICAgYWN0aW9uTmFtZTogJ0FjdGlvbicsXG4gICAgICB0ZW1wbGF0ZVBhdGg6IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoJ1Rlc3RBcnRpZmFjdCcpLmF0UGF0aCgnc29tZS9maWxlJyksXG4gICAgICBzdGFja05hbWU6ICdNeVN0YWNrJyxcbiAgICAgIGFkbWluUGVybWlzc2lvbnM6IGZhbHNlLFxuICAgICAgcmVwbGFjZU9uRmFpbHVyZTogdHJ1ZSxcbiAgICB9KTtcbiAgICBuZXcgU3RhZ2VEb3VibGUoe1xuICAgICAgcGlwZWxpbmU6IG5ldyBQaXBlbGluZURvdWJsZShzdGFjaywgJ1BpcGVsaW5lJywgeyByb2xlOiBwaXBlbGluZVJvbGUgfSksXG4gICAgICBhY3Rpb25zOiBbYWN0aW9uXSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGFja0FybiA9IF9zdGFja0FybignTXlTdGFjaycsIHN0YWNrKTtcblxuICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrKicsIHN0YWNrQXJuKTtcbiAgICBfYXNzZXJ0UGVybWlzc2lvbkdyYW50ZWQodGVzdCwgc3RhY2ssIHBpcGVsaW5lUm9sZS5zdGF0ZW1lbnRzLCAnY2xvdWRmb3JtYXRpb246Q3JlYXRlU3RhY2snLCBzdGFja0Fybik7XG4gICAgX2Fzc2VydFBlcm1pc3Npb25HcmFudGVkKHRlc3QsIHN0YWNrLCBwaXBlbGluZVJvbGUuc3RhdGVtZW50cywgJ2Nsb3VkZm9ybWF0aW9uOlVwZGF0ZVN0YWNrJywgc3RhY2tBcm4pO1xuICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdjbG91ZGZvcm1hdGlvbjpEZWxldGVTdGFjaycsIHN0YWNrQXJuKTtcblxuICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdpYW06UGFzc1JvbGUnLCBhY3Rpb24uZGVwbG95bWVudFJvbGUucm9sZUFybik7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGhlIERlbGV0ZVN0YWNrIEFjdGlvbiBzZXRzIHRoZSBEZXNjcmliZVN0YWNrKiwgRGVsZXRlU3RhY2sgJiBQYXNzUm9sZSBwZXJtaXNzaW9ucycodGVzdDogbm9kZXVuaXQuVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHBpcGVsaW5lUm9sZSA9IG5ldyBSb2xlRG91YmxlKHN0YWNrLCAnUGlwZWxpbmVSb2xlJyk7XG4gICAgY29uc3QgYWN0aW9uID0gbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkRlbGV0ZVN0YWNrQWN0aW9uKHtcbiAgICAgIGFjdGlvbk5hbWU6ICdBY3Rpb24nLFxuICAgICAgYWRtaW5QZXJtaXNzaW9uczogZmFsc2UsXG4gICAgICBzdGFja05hbWU6ICdNeVN0YWNrJyxcbiAgICB9KTtcbiAgICBuZXcgU3RhZ2VEb3VibGUoe1xuICAgICAgcGlwZWxpbmU6IG5ldyBQaXBlbGluZURvdWJsZShzdGFjaywgJ1BpcGVsaW5lJywgeyByb2xlOiBwaXBlbGluZVJvbGUgfSksXG4gICAgICBhY3Rpb25zOiBbYWN0aW9uXSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGFja0FybiA9IF9zdGFja0FybignTXlTdGFjaycsIHN0YWNrKTtcblxuICAgIF9hc3NlcnRQZXJtaXNzaW9uR3JhbnRlZCh0ZXN0LCBzdGFjaywgcGlwZWxpbmVSb2xlLnN0YXRlbWVudHMsICdjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrKicsIHN0YWNrQXJuKTtcbiAgICBfYXNzZXJ0UGVybWlzc2lvbkdyYW50ZWQodGVzdCwgc3RhY2ssIHBpcGVsaW5lUm9sZS5zdGF0ZW1lbnRzLCAnY2xvdWRmb3JtYXRpb246RGVsZXRlU3RhY2snLCBzdGFja0Fybik7XG5cbiAgICBfYXNzZXJ0UGVybWlzc2lvbkdyYW50ZWQodGVzdCwgc3RhY2ssIHBpcGVsaW5lUm9sZS5zdGF0ZW1lbnRzLCAnaWFtOlBhc3NSb2xlJywgYWN0aW9uLmRlcGxveW1lbnRSb2xlLnJvbGVBcm4pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59KTtcblxuaW50ZXJmYWNlIFBvbGljeVN0YXRlbWVudEpzb24ge1xuICBFZmZlY3Q6ICdBbGxvdycgfCAnRGVueSc7XG4gIEFjdGlvbjogc3RyaW5nIHwgc3RyaW5nW107XG4gIFJlc291cmNlOiBzdHJpbmcgfMKgc3RyaW5nW107XG4gIENvbmRpdGlvbjogYW55O1xufVxuXG5mdW5jdGlvbiBfYXNzZXJ0QWN0aW9uTWF0Y2hlcyhcbiAgdGVzdDogbm9kZXVuaXQuVGVzdCxcbiAgc3RhY2s6IGNkay5TdGFjayxcbiAgYWN0aW9uczogRnVsbEFjdGlvbltdLFxuICBwcm92aWRlcjogc3RyaW5nLFxuICBjYXRlZ29yeTogc3RyaW5nLFxuICBjb25maWd1cmF0aW9uPzogeyBba2V5OiBzdHJpbmddOiBhbnkgfSkge1xuICBjb25zdCBjb25maWd1cmF0aW9uU3RyID0gY29uZmlndXJhdGlvblxuICAgID8gYCwgY29uZmlndXJhdGlvbiBpbmNsdWRpbmcgJHtKU09OLnN0cmluZ2lmeShzdGFjay5yZXNvbHZlKGNvbmZpZ3VyYXRpb24pLCBudWxsLCAyKX1gXG4gICAgOiAnJztcbiAgY29uc3QgYWN0aW9uc1N0ciA9IEpTT04uc3RyaW5naWZ5KGFjdGlvbnMubWFwKGEgPT5cbiAgICAoeyBvd25lcjogYS5hY3Rpb25Qcm9wZXJ0aWVzLm93bmVyLCBwcm92aWRlcjogYS5hY3Rpb25Qcm9wZXJ0aWVzLnByb3ZpZGVyLFxuICAgICAgY2F0ZWdvcnk6IGEuYWN0aW9uUHJvcGVydGllcy5jYXRlZ29yeSwgY29uZmlndXJhdGlvbjogc3RhY2sucmVzb2x2ZShhLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uKSxcbiAgICB9KSxcbiAgKSwgbnVsbCwgMik7XG4gIHRlc3Qub2soX2hhc0FjdGlvbihzdGFjaywgYWN0aW9ucywgcHJvdmlkZXIsIGNhdGVnb3J5LCBjb25maWd1cmF0aW9uKSxcbiAgICBgRXhwZWN0ZWQgdG8gZmluZCBhbiBhY3Rpb24gd2l0aCBwcm92aWRlciAke3Byb3ZpZGVyfSwgY2F0ZWdvcnkgJHtjYXRlZ29yeX0ke2NvbmZpZ3VyYXRpb25TdHJ9LCBidXQgZm91bmQgJHthY3Rpb25zU3RyfWApO1xufVxuXG5mdW5jdGlvbiBfaGFzQWN0aW9uKFxuICBzdGFjazogY2RrLlN0YWNrLCBhY3Rpb25zOiBGdWxsQWN0aW9uW10sIHByb3ZpZGVyOiBzdHJpbmcsIGNhdGVnb3J5OiBzdHJpbmcsXG4gIGNvbmZpZ3VyYXRpb24/OiB7IFtrZXk6IHN0cmluZ106IGFueX0pIHtcbiAgZm9yIChjb25zdCBhY3Rpb24gb2YgYWN0aW9ucykge1xuICAgIGlmIChhY3Rpb24uYWN0aW9uUHJvcGVydGllcy5wcm92aWRlciAhPT0gcHJvdmlkZXIpwqB7IGNvbnRpbnVlOyB9XG4gICAgaWYgKGFjdGlvbi5hY3Rpb25Qcm9wZXJ0aWVzLmNhdGVnb3J5ICE9PSBjYXRlZ29yeSnCoHsgY29udGludWU7IH1cbiAgICBpZiAoY29uZmlndXJhdGlvbiAmJiAhYWN0aW9uLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uKSB7IGNvbnRpbnVlOyB9XG4gICAgaWYgKGNvbmZpZ3VyYXRpb24pIHtcbiAgICAgIGZvciAoY29uc3Qga2V5IG9mIE9iamVjdC5rZXlzKGNvbmZpZ3VyYXRpb24pKSB7XG4gICAgICAgIGlmICghXy5pc0VxdWFsKHN0YWNrLnJlc29sdmUoYWN0aW9uLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uW2tleV0pLCBzdGFjay5yZXNvbHZlKGNvbmZpZ3VyYXRpb25ba2V5XSkpKSB7XG4gICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cbiAgcmV0dXJuIGZhbHNlO1xufVxuXG5mdW5jdGlvbiBfYXNzZXJ0UGVybWlzc2lvbkdyYW50ZWQoXG4gIHRlc3Q6IG5vZGV1bml0LlRlc3QsXG4gIHN0YWNrOiBjZGsuU3RhY2ssXG4gIHN0YXRlbWVudHM6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSxcbiAgYWN0aW9uOiBzdHJpbmcsXG4gIHJlc291cmNlOiBzdHJpbmcsXG4gIGNvbmRpdGlvbnM/OiBhbnkpIHtcbiAgY29uc3QgY29uZGl0aW9uU3RyID0gY29uZGl0aW9uc1xuICAgID8gYCB3aXRoIGNvbmRpdGlvbihzKSAke0pTT04uc3RyaW5naWZ5KHN0YWNrLnJlc29sdmUoY29uZGl0aW9ucykpfWBcbiAgICA6ICcnO1xuICBjb25zdCByZXNvbHZlZFN0YXRlbWVudHMgPSBzdGFjay5yZXNvbHZlKHN0YXRlbWVudHMubWFwKHMgPT4gcy50b1N0YXRlbWVudEpzb24oKSkpO1xuICBjb25zdCBzdGF0ZW1lbnRzU3RyID0gSlNPTi5zdHJpbmdpZnkocmVzb2x2ZWRTdGF0ZW1lbnRzLCBudWxsLCAyKTtcbiAgdGVzdC5vayhfZ3JhbnRzUGVybWlzc2lvbihzdGFjaywgcmVzb2x2ZWRTdGF0ZW1lbnRzLCBhY3Rpb24sIHJlc291cmNlLCBjb25kaXRpb25zKSxcbiAgICBgRXhwZWN0ZWQgdG8gZmluZCBhIHN0YXRlbWVudCBncmFudGluZyAke2FjdGlvbn0gb24gJHtKU09OLnN0cmluZ2lmeShzdGFjay5yZXNvbHZlKHJlc291cmNlKSl9JHtjb25kaXRpb25TdHJ9LCBmb3VuZDpcXG4ke3N0YXRlbWVudHNTdHJ9YCk7XG59XG5cbmZ1bmN0aW9uIF9ncmFudHNQZXJtaXNzaW9uKHN0YWNrOiBjZGsuU3RhY2ssIHN0YXRlbWVudHM6IFBvbGljeVN0YXRlbWVudEpzb25bXSwgYWN0aW9uOiBzdHJpbmcsIHJlc291cmNlOiBzdHJpbmcsIGNvbmRpdGlvbnM/OiBhbnkpIHtcbiAgZm9yIChjb25zdCBzdGF0ZW1lbnQgb2Ygc3RhdGVtZW50cy5maWx0ZXIocyA9PiBzLkVmZmVjdCA9PT0gJ0FsbG93JykpIHtcbiAgICBpZiAoIV9pc09yQ29udGFpbnMoc3RhY2ssIHN0YXRlbWVudC5BY3Rpb24sIGFjdGlvbikpIHsgY29udGludWU7IH1cbiAgICBpZiAoIV9pc09yQ29udGFpbnMoc3RhY2ssIHN0YXRlbWVudC5SZXNvdXJjZSwgcmVzb3VyY2UpKSB7IGNvbnRpbnVlOyB9XG4gICAgaWYgKGNvbmRpdGlvbnMgJiYgIV9pc09yQ29udGFpbnMoc3RhY2ssIHN0YXRlbWVudC5Db25kaXRpb24sIGNvbmRpdGlvbnMpKSB7IGNvbnRpbnVlOyB9XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cbiAgcmV0dXJuIGZhbHNlO1xufVxuXG5mdW5jdGlvbiBfaXNPckNvbnRhaW5zKHN0YWNrOiBjZGsuU3RhY2ssIGVudGl0eTogc3RyaW5nIHwgc3RyaW5nW10sIHZhbHVlOiBzdHJpbmcpOiBib29sZWFuIHtcbiAgY29uc3QgcmVzb2x2ZWRWYWx1ZSA9IHN0YWNrLnJlc29sdmUodmFsdWUpO1xuICBjb25zdCByZXNvbHZlZEVudGl0eSA9IHN0YWNrLnJlc29sdmUoZW50aXR5KTtcbiAgaWYgKF8uaXNFcXVhbChyZXNvbHZlZEVudGl0eSwgcmVzb2x2ZWRWYWx1ZSkpIHsgcmV0dXJuIHRydWU7IH1cbiAgaWYgKCFBcnJheS5pc0FycmF5KHJlc29sdmVkRW50aXR5KSkgeyByZXR1cm4gZmFsc2U7IH1cbiAgZm9yIChjb25zdCB0ZXN0ZWQgb2YgZW50aXR5KSB7XG4gICAgaWYgKF8uaXNFcXVhbCh0ZXN0ZWQsIHJlc29sdmVkVmFsdWUpKSB7IHJldHVybiB0cnVlOyB9XG4gIH1cbiAgcmV0dXJuIGZhbHNlO1xufVxuXG5mdW5jdGlvbiBfc3RhY2tBcm4oc3RhY2tOYW1lOiBzdHJpbmcsIHNjb3BlOiBjZGsuSUNvbnN0cnVjdCk6IHN0cmluZyB7XG4gIHJldHVybiBjZGsuU3RhY2sub2Yoc2NvcGUpLmZvcm1hdEFybih7XG4gICAgc2VydmljZTogJ2Nsb3VkZm9ybWF0aW9uJyxcbiAgICByZXNvdXJjZTogJ3N0YWNrJyxcbiAgICByZXNvdXJjZU5hbWU6IGAke3N0YWNrTmFtZX0vKmAsXG4gIH0pO1xufVxuXG5jbGFzcyBQaXBlbGluZURvdWJsZSBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIGNvZGVwaXBlbGluZS5JUGlwZWxpbmUge1xuICBwdWJsaWMgcmVhZG9ubHkgcGlwZWxpbmVOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBwaXBlbGluZUFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcm9sZTogaWFtLlJvbGU7XG4gIHB1YmxpYyByZWFkb25seSBhcnRpZmFjdEJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgeyBwaXBlbGluZU5hbWUsIHJvbGUgfTogeyBwaXBlbGluZU5hbWU/OiBzdHJpbmcsIHJvbGU6IGlhbS5Sb2xlIH0pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIHRoaXMucGlwZWxpbmVOYW1lID0gcGlwZWxpbmVOYW1lIHx8ICdUZXN0UGlwZWxpbmUnO1xuICAgIHRoaXMucGlwZWxpbmVBcm4gPSBjZGsuU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHsgc2VydmljZTogJ2NvZGVwaXBlbGluZScsIHJlc291cmNlOiAncGlwZWxpbmUnLCByZXNvdXJjZU5hbWU6IHRoaXMucGlwZWxpbmVOYW1lIH0pO1xuICAgIHRoaXMucm9sZSA9IHJvbGU7XG4gICAgdGhpcy5hcnRpZmFjdEJ1Y2tldCA9IG5ldyBCdWNrZXREb3VibGUoc2NvcGUsICdCdWNrZXREb3VibGUnKTtcbiAgfVxuXG4gIHB1YmxpYyBvbkV2ZW50KF9pZDogc3RyaW5nLCBfb3B0aW9uczogZXZlbnRzLk9uRXZlbnRPcHRpb25zKTogZXZlbnRzLlJ1bGUge1xuICAgIHRocm93IG5ldyBFcnJvcignTWV0aG9kIG5vdCBpbXBsZW1lbnRlZC4nKTtcbiAgfVxuICBwdWJsaWMgb25TdGF0ZUNoYW5nZShfaWQ6IHN0cmluZywgX29wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ01ldGhvZCBub3QgaW1wbGVtZW50ZWQuJyk7XG4gIH1cbn1cblxuY2xhc3MgRnVsbEFjdGlvbiB7XG4gIGNvbnN0cnVjdG9yKFxuICAgIHJlYWRvbmx5IGFjdGlvblByb3BlcnRpZXM6IGNvZGVwaXBlbGluZS5BY3Rpb25Qcm9wZXJ0aWVzLFxuICAgIHJlYWRvbmx5IGFjdGlvbkNvbmZpZzogY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZykge1xuICAgIC8vIGVtcHR5XG4gIH1cbn1cblxuY2xhc3MgU3RhZ2VEb3VibGUgaW1wbGVtZW50cyBjb2RlcGlwZWxpbmUuSVN0YWdlIHtcbiAgcHVibGljIHJlYWRvbmx5IHN0YWdlTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcGlwZWxpbmU6IGNvZGVwaXBlbGluZS5JUGlwZWxpbmU7XG4gIHB1YmxpYyByZWFkb25seSBhY3Rpb25zOiBjb2RlcGlwZWxpbmUuSUFjdGlvbltdID0gW107XG4gIHB1YmxpYyByZWFkb25seSBmdWxsQWN0aW9uczogRnVsbEFjdGlvbltdO1xuXG4gIHB1YmxpYyBnZXQgbm9kZSgpOiBjZGsuQ29uc3RydWN0Tm9kZSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdTdGFnZURvdWJsZSBpcyBub3QgYSByZWFsIGNvbnN0cnVjdCcpO1xuICB9XG5cbiAgY29uc3RydWN0b3IoeyBuYW1lLCBwaXBlbGluZSwgYWN0aW9ucyB9OiB7IG5hbWU/OiBzdHJpbmcsIHBpcGVsaW5lOiBQaXBlbGluZURvdWJsZSwgYWN0aW9uczogY29kZXBpcGVsaW5lLklBY3Rpb25bXSB9KSB7XG4gICAgdGhpcy5zdGFnZU5hbWUgPSBuYW1lIHx8ICdUZXN0U3RhZ2UnO1xuICAgIHRoaXMucGlwZWxpbmUgPSBwaXBlbGluZTtcblxuICAgIGNvbnN0IHN0YWdlUGFyZW50ID0gbmV3IGNkay5Db25zdHJ1Y3QocGlwZWxpbmUsIHRoaXMuc3RhZ2VOYW1lKTtcbiAgICBjb25zdCBmdWxsQWN0aW9ucyA9IG5ldyBBcnJheTxGdWxsQWN0aW9uPigpO1xuICAgIGZvciAoY29uc3QgYWN0aW9uIG9mIGFjdGlvbnMpIHtcbiAgICAgIGNvbnN0IGFjdGlvblBhcmVudCA9IG5ldyBjZGsuQ29uc3RydWN0KHN0YWdlUGFyZW50LCBhY3Rpb24uYWN0aW9uUHJvcGVydGllcy5hY3Rpb25OYW1lKTtcbiAgICAgIGZ1bGxBY3Rpb25zLnB1c2gobmV3IEZ1bGxBY3Rpb24oYWN0aW9uLmFjdGlvblByb3BlcnRpZXMsIGFjdGlvbi5iaW5kKGFjdGlvblBhcmVudCwgdGhpcywge1xuICAgICAgICByb2xlOiBwaXBlbGluZS5yb2xlLFxuICAgICAgICBidWNrZXQ6IHBpcGVsaW5lLmFydGlmYWN0QnVja2V0LFxuICAgICAgfSkpKTtcbiAgICB9XG4gICAgdGhpcy5mdWxsQWN0aW9ucyA9IGZ1bGxBY3Rpb25zO1xuICB9XG5cbiAgcHVibGljIGFkZEFjdGlvbihfYWN0aW9uOiBjb2RlcGlwZWxpbmUuSUFjdGlvbik6IHZvaWQge1xuICAgIHRocm93IG5ldyBFcnJvcignYWRkQWN0aW9uKCkgaXMgbm90IHN1cHBvcnRlZCBvbiBTdGFnZURvdWJsZScpO1xuICB9XG5cbiAgcHVibGljIG9uU3RhdGVDaGFuZ2UoX25hbWU6IHN0cmluZywgX3RhcmdldD86IGV2ZW50cy5JUnVsZVRhcmdldCwgX29wdGlvbnM/OiBldmVudHMuUnVsZVByb3BzKTpcbiAgZXZlbnRzLlJ1bGUge1xuICAgIHRocm93IG5ldyBFcnJvcignb25TdGF0ZUNoYW5nZSgpIGlzIG5vdCBzdXBwb3J0ZWQgb24gU3RhZ2VEb3VibGUnKTtcbiAgfVxufVxuXG5jbGFzcyBSb2xlRG91YmxlIGV4dGVuZHMgaWFtLlJvbGUge1xuICBwdWJsaWMgcmVhZG9ubHkgc3RhdGVtZW50cyA9IG5ldyBBcnJheTxpYW0uUG9saWN5U3RhdGVtZW50PigpO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogaWFtLlJvbGVQcm9wcyA9IHsgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3Rlc3QnKSB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gIH1cblxuICBwdWJsaWMgYWRkVG9Qb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogYm9vbGVhbiB7XG4gICAgc3VwZXIuYWRkVG9Qb2xpY3koc3RhdGVtZW50KTtcbiAgICB0aGlzLnN0YXRlbWVudHMucHVzaChzdGF0ZW1lbnQpO1xuICAgIHJldHVybiB0cnVlO1xuICB9XG59XG5cbmNsYXNzIEJ1Y2tldERvdWJsZSBleHRlbmRzIHMzLkJ1Y2tldCB7XG4gIHB1YmxpYyBncmFudFJlYWQoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlLCBfb2JqZWN0c0tleVBhdHRlcm46IGFueSA9ICcqJyk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5kcm9wKGlkZW50aXR5LCAnJyk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRXcml0ZShpZGVudGl0eTogaWFtLklHcmFudGFibGUsIF9vYmplY3RzS2V5UGF0dGVybjogYW55ID0gJyonKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmRyb3AoaWRlbnRpdHksICcnKTtcbiAgfVxuXG4gIHB1YmxpYyBncmFudFJlYWRXcml0ZShpZGVudGl0eTogaWFtLklHcmFudGFibGUsIF9vYmplY3RzS2V5UGF0dGVybjogYW55ID0gJyonKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmRyb3AoaWRlbnRpdHksICcnKTtcbiAgfVxufVxuIl19