"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const stepfunctions = require("@aws-cdk/aws-stepfunctions");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with new Lambda function
// --------------------------------------------------------------
test('Test deployment with new Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    new lib_1.LambdaToStepFunction(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'deploy-function'
            }
        },
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'deploy-function'
            }
        }
    });
});
// --------------------------------------------------------------
// Test deployment with existing Lambda function
// --------------------------------------------------------------
test('Test deployment with existing Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        environment: {
            LAMBDA_NAME: 'existing-function'
        }
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    // Add the pattern
    new lib_1.LambdaToStepFunction(stack, 'test-lambda-step-function-stack', {
        existingLambdaObj: fn,
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'existing-function'
            }
        }
    });
});
// --------------------------------------------------------------
// Test invocation permissions
// --------------------------------------------------------------
test('Test invocation permissions', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        environment: {
            LAMBDA_NAME: 'existing-function'
        }
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    // Add the pattern
    new lib_1.LambdaToStepFunction(stack, 'test-lambda-step-function-stack', {
        existingLambdaObj: fn,
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testlambdastepfunctionstackStateMachine373C0BB9"
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
// --------------------------------------------------------------
// Test the properties
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const pattern = new lib_1.LambdaToStepFunction(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'existing-function'
            }
        },
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func !== null);
    // Assertion 2
    const stateMachine = pattern.stateMachine;
    expect(stateMachine !== null);
    // Assertion 3
    const cwAlarm = pattern.cloudwatchAlarms;
    expect(cwAlarm !== null);
    expect(pattern.stateMachineLogGroup !== null);
});
// --------------------------------------------------------------
// Test the properties
// --------------------------------------------------------------
test('Test the properties with no CW Alarms', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const pattern = new lib_1.LambdaToStepFunction(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'existing-function'
            }
        },
        stateMachineProps: {
            definition: startState
        },
        createCloudWatchAlarms: false
    });
    // Assertion 1
    expect(pattern.lambdaFunction !== null);
    // Assertion 2
    expect(pattern.stateMachine !== null);
    // Assertion 3
    expect(pattern.cloudwatchAlarms === null);
    expect(pattern.stateMachineLogGroup !== null);
});
//# sourceMappingURL=data:application/json;base64,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