[CmdletBinding()]
param(
    [Parameter(Mandatory=$true)][string]$SourcesRef,
    [Parameter(Mandatory=$false)][string]$PortDirectory = $PSScriptRoot,
    [Parameter(Mandatory=$false)][string]$vcpkg = "$PSScriptRoot/../../vcpkg"
)

$ErrorActionPreference = "Stop"

$ManifestIn = "$PortDirectory/vcpkg.in.json"
$ManifestOut = "$PortDirectory/vcpkg.json"
$CMakeFragmentFile = "$PortDirectory/compute_build_only.cmake"

$ExtractedSources = "${env:TEMP}/aws-sdk-cpp-generateFeatures-$SourcesRef"
if (-not (Test-Path $ExtractedSources)) {
    if (Test-Path "$ExtractedSources.tmp") {
        Remove-Item -Force "$ExtractedSources.tmp"
    }
    git clone "https://github.com/aws/aws-sdk-cpp" "$ExtractedSources.tmp" | Out-Host
    git -c "$ExtractedSources.tmp" checkout $SourcesRef
    Move-Item "$ExtractedSources.tmp" "$ExtractedSources"
}
Write-Host "Using sources directory: $ExtractedSources"


$subfolders = Get-Item $ExtractedSources\aws-cpp-sdk-*

$manifest = Get-Content $ManifestIn | ConvertFrom-Json
$manifest | Add-Member `
    -NotePropertyName '$note' `
    -NotePropertyValue 'Automatically generated by generateFeatures.ps1'
$manifest | Add-Member -NotePropertyName 'features' -NotePropertyValue @{}

$cmakefragmenttext = @("# Automatically generated by generateFeatures.ps1")

function GetDescription($dir, $modulename)
{
    if (Test-Path "$dir\CMakeLists.txt")
    {
        $descs = @(Select-String -Path "$dir\CMakeLists.txt" -Pattern "`"C\+\+ SDK for the AWS [^`"]*`"")
        if ($descs.count -eq 1) {
            $desc = $descs[0].Matches.Value -replace "`"",""
            "$desc"
        }
        else { "C++ SDK for the AWS $modulename service" }
    }
    else { "C++ SDK for the AWS $modulename service" }
}

foreach ($subfolder in $subfolders)
{
    $modulename = $subfolder.name -replace "^aws-cpp-sdk-",""
    if ($modulename -match "-tests`$") { continue }
    if ($modulename -match "-sample`$") { continue }
    if ($modulename -eq "core") { continue }

    $lowermodulename = $modulename.ToLower()

    $manifest.features.Add("$lowermodulename", @{ description=(GetDescription $subfolder $modulename) })

    $cmakefragmenttext += @(
        "if(`"$lowermodulename`" IN_LIST FEATURES)",
        "  list(APPEND BUILD_ONLY $modulename)",
        "endif()"
    )
}

[IO.File]::WriteAllText($ManifestOut, (ConvertTo-Json -Depth 5 -InputObject $manifest))

Write-Verbose ($cmakefragmenttext -join "`n")
[IO.File]::WriteAllText($CMakeFragmentFile, ($cmakefragmenttext -join "`n") +"`n")

& $vcpkg format-manifest --feature-flags=-manifests $ManifestOut
