# -*- coding: utf-8 -*-
"""
      ┏┓       ┏┓
    ┏━┛┻━━━━━━━┛┻━┓
    ┃      ☃      ┃
    ┃  ┳┛     ┗┳  ┃
    ┃      ┻      ┃
    ┗━┓         ┏━┛
      ┗┳        ┗━┓
       ┃          ┣┓
       ┃          ┏┛
       ┗┓┓┏━━━━┳┓┏┛
        ┃┫┫    ┃┫┫
        ┗┻┛    ┗┻┛
    God Bless,Never Bug
"""
import json
from flask import request
from inspect import signature
from functools import wraps

from memoization import cached


class ApiCache:
    def __init__(self, redis=None, key_func=None, expired_time=24 * 60 * 60):
        """
            # flask-api-cache
            **A package for caching flask api with custom key.**


            ## Description:
            A decorator for python flask api.

            You can set cache in your **memory** or with **redis instance**,<br>
            the key will be generated by the following rule:<br>
            `{YOUR_FUNCTION_NAME}:{REQUEST_FULL_PATH}`<br>
            or you can use your custom key function by key_func args,<br>
            the value will be your function return value.<br>

            ## How To Use:
            ### Cache Without Redis
            ```python
            @app.route('/')
            @ApiCache(expired_time=10)
            def index(*args, **kwargs):
                name = request.args.get('name')
                age = request.args.get('age')
                return f'{name} is {age} years old.'
            ```
            If you request for **http://0.0.0.0:5000?name=Jimmy&age=18**,<br>
            it will set a 10 seconds cache by key: `name=Jimmy&age=18`,<br>
                                 with value: `Jimmy is 18 years old.`,
            in your memory, it will be cleared after api service restart.

            ### Cache With Redis
            ```python
            @app.route('/')
            @ApiCache(redis=REDIS_INSTANCE, expired_time=10)
            def index(*args, **kwargs):
                name = request.args.get('name')
                age = request.args.get('age')
                return f'{name} is {age} years old.'
            ```
            If you request for **http://0.0.0.0:5000?name=Jimmy&age=18** ,<br>
            it will set a 10 seconds cache by key: `name=Jimmy&age=18`,<br>
                                 with value: `Jimmy is 18 years old.`,
            in your redis instance.

            ### Cache With Custom Function
            ```python
            def custom_func(**kwargs):
                name = kwargs.get('name')
                age = kwargs.get('age')
                sex = kwargs.get('sex')
                keys = f'{name}:{age}:{sex}'
                return keys

            @app.route('/<string:name>/<int:age>')
            @ApiCache(key_func=custom_func, expired_time=10)
            def index(name, age):
                sex = request.args.get('sex', None, str)
                return f'{name} is a {age} years old {sex}.'
            ```
            If you request for **http://0.0.0.0:5600/Jimmy/18?sex=boy**,<br>
            it will set a 10 seconds cache by key: `Jimmy:18:boy`,<br>
                                 with value: `Jimmy is a 18 years old boy.`,
            in your memory, it will be cleared after service restart.

            ## Parameters

            |name|required|description|
            |----|--------|-----------|
            |redis|no|if you want to caching data in redis, you can call ApiCache with a redis instance.|
            |expired_time|no|set your expired time with **seconds**, the default value is 24 * 60 * 60 seconds (1 day)|
            |key_func|no|the function which you want to make custom key|
        """
        self.redis = redis
        self.key_func = key_func
        self.expired_time = expired_time
        self.func = None
        self.args = None
        self.kwargs = None
        self.func_args_dict = dict()
        self._valid_redis()

    def __call__(self, func):
        self.func = func
        if self.redis:
            return self._cache_in_redis()
        else:
            return self._cache_in_memory()

    def _valid_redis(self):
        """
        check if redis service is available for connect
        """
        if not self.redis:
            return
        try:
            self.redis.ping()
        except Exception as e:
            raise Exception('redis server not available')

    def _get_redis_key(self):
        """
        generate redis key default by function name and request full path,
        or you can use your custom function
        """
        if self.key_func:
            self.func_args_dict.update(self.kwargs)
            self.func_args_dict.update(request.args.to_dict())
            return self.key_func(**self.func_args_dict)
        else:
            return f'{self.func.__name__}:{request.full_path}'

    def _get_redis_value(self, key):
        """
        get redis value by key
        """
        return self.redis.get(name=key)

    def _set_redis_value(self, key, value):
        """
        set redis
        """
        self.redis.set(name=key, value=value, ex=self.expired_time)

    @staticmethod
    def _update_signature(params):
        """
        update input function signature
        """

        def outer_wrapper(f):
            @wraps(f)
            def wrapper(*args, **kwargs):
                return f()

            # override signature
            sign = signature(f)
            sign = sign.replace(parameters=tuple(params))
            wrapper.__signature__ = sign
            return wrapper

        return outer_wrapper

    def _cache_in_memory(self):
        @self._update_signature(signature(self.func).parameters.values())
        def make_custom_key(*args):
            return self._get_redis_key()

        f = cached(ttl=self.expired_time, custom_key_maker=make_custom_key)(self.func)

        @wraps(f)
        def wrapper(*args, **kwargs):
            self.args = args
            self.kwargs = kwargs
            return f(*args, **kwargs)

        return wrapper

    def _cache_in_redis(self):
        def wrapper(*args, **kwargs):
            self.args = args
            self.kwargs = kwargs
            key = self._get_redis_key()
            value = self._get_redis_value(key=key)
            if not value:
                value = self.func(*args, **kwargs)
                value = json.dumps(value)
                self._set_redis_value(key=key, value=value)
            return value

        return wrapper
