"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientSettings = void 0;
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
// initialize logger
const logger = log4js_1.getLogger();
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ClientSettings extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.options = options;
        this.clientSettingsFilename =
            options.clientSettingsFilename ?? ".client-settings.json";
        this.clientSettingsPath = path.join(options.generatedCodeDir, this.clientSettingsFilename);
        // load previously generated client config
        const clientSettingsPathAbs = path.join(project.outdir, this.clientSettingsPath);
        logger.trace(`Reading client settings from ${clientSettingsPathAbs}`);
        const existingClientConfig = util_1.tryReadFileSync(clientSettingsPathAbs);
        const prevClientLanguages = new Set();
        const prevDocFormats = new Set();
        if (existingClientConfig) {
            const parsedClientConfig = JSON.parse(existingClientConfig);
            // previously generated client settings
            parsedClientConfig.clientLanguages.map((l) => prevClientLanguages.add(l));
            parsedClientConfig.documentationFormats.map((d) => prevDocFormats.add(d));
        }
        this.clientLanguageConfigs = options.clientLanguages.map((clientLanguage) => ({
            clientLanguage,
            isDefault: clientLanguage === options.defaultClientLanguage,
            generate: options.specChanged ||
                options.forceGenerateCodeAndDocs ||
                !prevClientLanguages.has(clientLanguage),
        }));
        this.documentationFormatConfigs = options.documentationFormats.map((documentationFormat) => ({
            documentationFormat,
            generate: options.specChanged ||
                options.forceGenerateCodeAndDocs ||
                !prevDocFormats.has(documentationFormat),
        }));
    }
    synthesize() {
        // store the client config
        // this helps to optimize to re-generate projects when new language/doc format is
        // introduced while the spec doesn't change
        logger.trace(`Generating settings file to ${this.clientSettingsPath}`);
        const clientSettingsJsonFile = new projen_1.JsonFile(this.project, this.clientSettingsPath, {
            obj: {
                clientLanguages: this.options.clientLanguages,
                documentationFormats: this.options.documentationFormats,
            },
            marker: true,
            editGitignore: false,
        });
        logger.debug(`Generated settings file to ${clientSettingsJsonFile.absolutePath}`);
    }
}
exports.ClientSettings = ClientSettings;
//# sourceMappingURL=data:application/json;base64,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