"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayLambdaApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const authorizers_1 = require("./authorizers");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const utils_1 = require("./spec/utils");
/**
 * A construct for creating an api gateway api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayLambdaApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                    },
                ],
            },
        ], true);
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        const prepareSpecOptions = {
            integrations: Object.fromEntries(Object.entries(integrations).map(([operation, integration]) => [
                operation,
                {
                    functionInvocationUri: utils_1.functionInvocationUri(this, integration.function),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(integration.authorizer || defaultAuthorizer || authorizers_1.Authorizers.none()),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const preparedSpec = prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(preparedSpec);
        // Grant API Gateway permission to invoke each lambda which implements an integration or custom authorizer
        api_gateway_integrations_1.getLabelledFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-IAM4",
                reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                appliesTo: [
                    {
                        regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                    },
                ],
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-APIG2",
                reason: "This construct implements fine grained validation via OpenApi.",
            },
        ], true);
    }
}
exports.OpenApiGatewayLambdaApi = OpenApiGatewayLambdaApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayLambdaApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayLambdaApi", version: "0.8.3" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3Blbi1hcGktZ2F0ZXdheS1sYW1iZGEtYXBpLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2NvbnN0cnVjdC9vcGVuLWFwaS1nYXRld2F5LWxhbWJkYS1hcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7Ozs7Ozs7Ozs7Ozs7d0hBY3dIO0FBRXhILDZCQUE2QjtBQUM3QiwwREFBc0Q7QUFDdEQsNkNBQThEO0FBQzlELCtEQVFvQztBQUNwQyxpREFNNkI7QUFDN0IsdURBS2dDO0FBQ2hDLG1EQUFnRDtBQUNoRCw2REFBa0Q7QUFDbEQsbUVBQXdEO0FBQ3hELHFDQUEwQztBQUMxQywyQ0FBdUM7QUFDdkMsK0NBQTRDO0FBRTVDLDRFQUdtRDtBQUVuRCw4REFHaUM7QUFDakMsOEVBQXVFO0FBQ3ZFLHdDQUFxRDtBQWtCckQ7O0dBRUc7QUFDSCxNQUFhLHVCQUF3QixTQUFRLHNCQUFTO0lBR3BELFlBQ0UsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLEtBQW1DO1FBRW5DLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxFQUNKLFlBQVksRUFDWixJQUFJLEVBQ0osUUFBUSxFQUNSLGVBQWUsRUFDZixpQkFBaUIsRUFDakIsV0FBVyxFQUNYLEdBQUcsT0FBTyxFQUNYLEdBQUcsS0FBSyxDQUFDO1FBRVYsb0NBQW9DO1FBQ3BDLE1BQU0sY0FBYyxHQUFHLElBQUkscUJBQUssQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO1lBQ2xELElBQUksRUFBRSxRQUFRO1NBQ2YsQ0FBQyxDQUFDO1FBQ0gsMERBQTBEO1FBQzFELE1BQU0sMkJBQTJCLEdBQUcsR0FBRyxjQUFjLENBQUMsV0FBVyxXQUFXLENBQUM7UUFFN0UsTUFBTSxLQUFLLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsTUFBTSxxQkFBcUIsR0FBRyxHQUFHLGdCQUFNLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQzthQUMxRCxLQUFLLENBQUMsR0FBRyxDQUFDO2FBQ1YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUM7UUFDMUIsTUFBTSxlQUFlLEdBQUcsSUFBSSxjQUFJLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3hELFNBQVMsRUFBRSxJQUFJLDBCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQ3ZELGNBQWMsRUFBRTtnQkFDZCxJQUFJLEVBQUUsSUFBSSx3QkFBYyxDQUFDO29CQUN2QixVQUFVLEVBQUU7d0JBQ1YsSUFBSSx5QkFBZSxDQUFDOzRCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLOzRCQUNwQixPQUFPLEVBQUU7Z0NBQ1AscUJBQXFCO2dDQUNyQixzQkFBc0I7Z0NBQ3RCLG1CQUFtQjs2QkFDcEI7NEJBQ0QsU0FBUyxFQUFFO2dDQUNULGdCQUFnQixLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLDBCQUEwQixxQkFBcUIsRUFBRTtnQ0FDOUYsZ0JBQWdCLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sMEJBQTBCLHFCQUFxQixJQUFJOzZCQUNqRzt5QkFDRixDQUFDO3FCQUNIO2lCQUNGLENBQUM7Z0JBQ0YsRUFBRSxFQUFFLElBQUksd0JBQWMsQ0FBQztvQkFDckIsVUFBVSxFQUFFO3dCQUNWLElBQUkseUJBQWUsQ0FBQzs0QkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSzs0QkFDcEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDOzRCQUN6QixTQUFTLEVBQUU7Z0NBQ1QsY0FBYyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQzs2QkFDaEU7eUJBQ0YsQ0FBQzt3QkFDRixJQUFJLHlCQUFlLENBQUM7NEJBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7NEJBQ3BCLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQzs0QkFDekIsU0FBUyxFQUFFO2dDQUNULHVHQUF1RztnQ0FDdkcsMEJBQTBCO2dDQUMxQixjQUFjLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FDakMsR0FBRywyQkFBMkIsSUFBSSxDQUNuQzs2QkFDRjt5QkFDRixDQUFDO3FCQUNIO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLGVBQWUsRUFDZjtZQUNFO2dCQUNFLEVBQUUsRUFBRSxtQkFBbUI7Z0JBQ3ZCLE1BQU0sRUFDSix1SUFBdUk7Z0JBQ3pJLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxLQUFLLEVBQUUsNEJBQTRCLGdCQUFNLENBQUMsbUJBQW1CLENBQzNELEtBQUssQ0FDTixJQUFJLGdCQUFNLENBQUMsb0JBQW9CLENBQzlCLEtBQUssQ0FDTiwwQkFBMEIscUJBQXFCLE9BQU87cUJBQ3hEO2lCQUNGO2FBQ0Y7U0FDRixFQUNELElBQUksQ0FDTCxDQUFDO1FBRUYseUdBQXlHO1FBQ3pHLE1BQU0sV0FBVyxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzFELE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7WUFDNUIsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxnREFBZ0QsQ0FBQyxDQUN2RTtZQUNELE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsSUFBSSxFQUFFLGVBQWU7WUFDckIsWUFBWSxFQUFFLHFCQUFxQjtTQUNwQyxDQUFDLENBQUM7UUFFSCxNQUFNLG9CQUFvQixHQUFHLEdBQUcscUJBQXFCLFdBQVcsQ0FBQztRQUNqRSxNQUFNLFlBQVksR0FBRyxJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUU7WUFDN0QsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDdkQsY0FBYyxFQUFFO2dCQUNkLElBQUksRUFBRSxJQUFJLHdCQUFjLENBQUM7b0JBQ3ZCLFVBQVUsRUFBRTt3QkFDVixJQUFJLHlCQUFlLENBQUM7NEJBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7NEJBQ3BCLE9BQU8sRUFBRTtnQ0FDUCxxQkFBcUI7Z0NBQ3JCLHNCQUFzQjtnQ0FDdEIsbUJBQW1COzZCQUNwQjs0QkFDRCxTQUFTLEVBQUU7Z0NBQ1QsZ0JBQWdCLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sMEJBQTBCLG9CQUFvQixFQUFFO2dDQUM3RixnQkFBZ0IsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTywwQkFBMEIsb0JBQW9CLElBQUk7NkJBQ2hHO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxRQUFRLEdBQUcsSUFBSSwyQkFBUSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUN6RCxjQUFjLEVBQUUsV0FBVztZQUMzQixJQUFJLEVBQUUsWUFBWTtZQUNsQixvQkFBb0I7U0FDckIsQ0FBQyxDQUFDO1FBRUgseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsWUFBWSxFQUNaO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUNKLHVJQUF1STthQUMxSTtTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFDRix5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxRQUFRLEVBQ1I7WUFDRTtnQkFDRSxFQUFFLEVBQUUsaUJBQWlCO2dCQUNyQixNQUFNLEVBQ0osbUdBQW1HO2FBQ3RHO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUVGLE1BQU0sa0JBQWtCLEdBQTBCO1lBQ2hELFlBQVksRUFBRSxNQUFNLENBQUMsV0FBVyxDQUM5QixNQUFNLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxFQUFFLEVBQUUsQ0FBQztnQkFDN0QsU0FBUztnQkFDVDtvQkFDRSxxQkFBcUIsRUFBRSw2QkFBcUIsQ0FDMUMsSUFBSSxFQUNKLFdBQVcsQ0FBQyxRQUFRLENBQ3JCO29CQUNELGdCQUFnQixFQUFFLGlEQUE4QixDQUM5QyxXQUFXLENBQUMsVUFBVSxJQUFJLGlCQUFpQixJQUFJLHlCQUFXLENBQUMsSUFBSSxFQUFFLENBQ2xFO2lCQUNGO2FBQ0YsQ0FBQyxDQUNIO1lBQ0QsZUFBZSxFQUFFLHlDQUFzQixDQUNyQyxJQUFJLEVBQ0osWUFBWSxFQUNaLGlCQUFpQixDQUNsQjtZQUNELFdBQVcsRUFBRSxXQUFXLElBQUk7Z0JBQzFCLFlBQVksRUFBRSxXQUFXLENBQUMsWUFBWSxJQUFJLHFCQUFJLENBQUMsZUFBZTtnQkFDOUQsWUFBWSxFQUFFLFdBQVcsQ0FBQyxZQUFZLElBQUkscUJBQUksQ0FBQyxXQUFXO2dCQUMxRCxZQUFZLEVBQUUsV0FBVyxDQUFDLFlBQVk7Z0JBQ3RDLFVBQVUsRUFBRSxXQUFXLENBQUMsVUFBVSxJQUFJLEdBQUc7YUFDMUM7WUFDRCxlQUFlO1NBQ2hCLENBQUM7UUFFRixnSEFBZ0g7UUFDaEgsaUdBQWlHO1FBQ2pHLE1BQU0sWUFBWSxHQUFHLDZCQUFjLENBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFFOUQsTUFBTSxzQ0FBc0MsR0FDMUM7WUFDRSxpQkFBaUIsRUFBRTtnQkFDakIsTUFBTSxFQUFFLGNBQWMsQ0FBQyxNQUFNLENBQUMsVUFBVTtnQkFDeEMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxXQUFXO2FBQ2hDO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLE1BQU0sRUFBRSxjQUFjLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQ3hDLEdBQUcsRUFBRSwyQkFBMkI7YUFDakM7WUFDRCxHQUFHLGtCQUFrQjtTQUN0QixDQUFDO1FBRUosTUFBTSx5QkFBeUIsR0FBRyxJQUFJLDRCQUFjLENBQ2xELElBQUksRUFDSiwyQkFBMkIsRUFDM0I7WUFDRSxZQUFZLEVBQUUsUUFBUSxDQUFDLFlBQVk7WUFDbkMsVUFBVSxFQUFFLHNDQUFzQztTQUNuRCxDQUNGLENBQUM7UUFFRixrSEFBa0g7UUFDbEgscUNBQXFDO1FBQ3JDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSw0QkFBVyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDbkMsYUFBYSxFQUFFLDhCQUFhLENBQUMsVUFBVSxDQUNyQyxjQUFjLENBQUMsTUFBTSxFQUNyQix5QkFBeUIsQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLENBQ3hEO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLG9CQUFvQixFQUFFLElBQUksdUNBQXNCLENBQzlDLElBQUksbUJBQVEsQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQ2xDO2dCQUNELGVBQWUsRUFBRSxnQ0FBZSxDQUFDLEdBQUcsRUFBRTtnQkFDdEMsWUFBWSxFQUFFLG1DQUFrQixDQUFDLElBQUk7YUFDdEM7WUFDRCxHQUFHLE9BQU87U0FDWCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUV2RCxvSEFBb0g7UUFDcEgsK0dBQStHO1FBQy9HLG9IQUFvSDtRQUNwSCxJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUV4RCwwR0FBMEc7UUFDMUcsK0NBQW9CLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxFQUFFLEVBQUU7WUFDbEUsSUFBSSwwQkFBYSxDQUFDLElBQUksRUFBRSxvQkFBb0IsS0FBSyxFQUFFLEVBQUU7Z0JBQ25ELE1BQU0sRUFBRSx1QkFBdUI7Z0JBQy9CLFNBQVMsRUFBRSwwQkFBMEI7Z0JBQ3JDLFlBQVksRUFBRSxNQUFNLENBQUMsV0FBVztnQkFDaEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3pCLE9BQU8sRUFBRSxhQUFhO29CQUN0QixRQUFRLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTO29CQUM1QixZQUFZLEVBQUUsS0FBSztpQkFDcEIsQ0FBQzthQUNILENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsSUFBSSxFQUNKO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUNKLDBFQUEwRTtnQkFDNUUsU0FBUyxFQUFFO29CQUNUO3dCQUNFLEtBQUssRUFBRSxpQkFBaUIsZ0JBQU0sQ0FBQyxzQkFBc0IsQ0FDbkQsS0FBSyxDQUNOLHVFQUF1RTtxQkFDekU7aUJBQ0Y7YUFDRjtTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFFRix5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxJQUFJLEVBQ0o7WUFDRTtnQkFDRSxFQUFFLEVBQUUsb0JBQW9CO2dCQUN4QixNQUFNLEVBQ0osZ0VBQWdFO2FBQ25FO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztJQUNKLENBQUM7O0FBNVJILDBEQTZSQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKipcbiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblxuIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuXG4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZSB3aXRoIHRoZSBMaWNlbnNlLlxuIFlvdSBtYXkgb2J0YWluIGEgY29weSBvZiB0aGUgTGljZW5zZSBhdFxuXG4gaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG5cbiBVbmxlc3MgcmVxdWlyZWQgYnkgYXBwbGljYWJsZSBsYXcgb3IgYWdyZWVkIHRvIGluIHdyaXRpbmcsIHNvZnR3YXJlXG4gZGlzdHJpYnV0ZWQgdW5kZXIgdGhlIExpY2Vuc2UgaXMgZGlzdHJpYnV0ZWQgb24gYW4gXCJBUyBJU1wiIEJBU0lTLFxuIFdJVEhPVVQgV0FSUkFOVElFUyBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBlaXRoZXIgZXhwcmVzcyBvciBpbXBsaWVkLlxuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9ucyBhbmRcbiBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKiAqL1xuXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBQREtOYWcgfSBmcm9tIFwiQGF3cy1wcm90b3R5cGluZy1zZGsvcGRrLW5hZ1wiO1xuaW1wb3J0IHsgQ3VzdG9tUmVzb3VyY2UsIER1cmF0aW9uLCBTdGFjayB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHtcbiAgQWNjZXNzTG9nRm9ybWF0LFxuICBBcGlEZWZpbml0aW9uLFxuICBDb3JzLFxuICBMb2dHcm91cExvZ0Rlc3RpbmF0aW9uLFxuICBNZXRob2RMb2dnaW5nTGV2ZWwsXG4gIFJlc3RBcGlCYXNlUHJvcHMsXG4gIFNwZWNSZXN0QXBpLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWFwaWdhdGV3YXlcIjtcbmltcG9ydCB7XG4gIEVmZmVjdCxcbiAgUG9saWN5RG9jdW1lbnQsXG4gIFBvbGljeVN0YXRlbWVudCxcbiAgUm9sZSxcbiAgU2VydmljZVByaW5jaXBhbCxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1pYW1cIjtcbmltcG9ydCB7XG4gIENmblBlcm1pc3Npb24sXG4gIENvZGUsXG4gIEZ1bmN0aW9uIGFzIExhbWJkYUZ1bmN0aW9uLFxuICBSdW50aW1lLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxhbWJkYVwiO1xuaW1wb3J0IHsgTG9nR3JvdXAgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxvZ3NcIjtcbmltcG9ydCB7IEFzc2V0IH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1zMy1hc3NldHNcIjtcbmltcG9ydCB7IFByb3ZpZGVyIH0gZnJvbSBcImF3cy1jZGstbGliL2N1c3RvbS1yZXNvdXJjZXNcIjtcbmltcG9ydCB7IE5hZ1N1cHByZXNzaW9ucyB9IGZyb20gXCJjZGstbmFnXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgQXV0aG9yaXplcnMgfSBmcm9tIFwiLi9hdXRob3JpemVyc1wiO1xuaW1wb3J0IHsgUHJlcGFyZUFwaVNwZWNDdXN0b21SZXNvdXJjZVByb3BlcnRpZXMgfSBmcm9tIFwiLi9wcmVwYXJlLXNwZWMtZXZlbnQtaGFuZGxlclwiO1xuaW1wb3J0IHtcbiAgcHJlcGFyZUFwaVNwZWMsXG4gIFByZXBhcmVBcGlTcGVjT3B0aW9ucyxcbn0gZnJvbSBcIi4vcHJlcGFyZS1zcGVjLWV2ZW50LWhhbmRsZXIvcHJlcGFyZS1zcGVjXCI7XG5pbXBvcnQgeyBPcGVuQXBpT3B0aW9ucyB9IGZyb20gXCIuL3NwZWNcIjtcbmltcG9ydCB7XG4gIHByZXBhcmVTZWN1cml0eVNjaGVtZXMsXG4gIHNlcmlhbGl6ZUFzQXV0aG9yaXplclJlZmVyZW5jZSxcbn0gZnJvbSBcIi4vc3BlYy9hcGktZ2F0ZXdheS1hdXRoXCI7XG5pbXBvcnQgeyBnZXRMYWJlbGxlZEZ1bmN0aW9ucyB9IGZyb20gXCIuL3NwZWMvYXBpLWdhdGV3YXktaW50ZWdyYXRpb25zXCI7XG5pbXBvcnQgeyBmdW5jdGlvbkludm9jYXRpb25VcmkgfSBmcm9tIFwiLi9zcGVjL3V0aWxzXCI7XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgdGhlIE9wZW5BcGlHYXRld2F5TGFtYmRhQXBpIGNvbnN0cnVjdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5BcGlHYXRld2F5TGFtYmRhQXBpUHJvcHNcbiAgZXh0ZW5kcyBSZXN0QXBpQmFzZVByb3BzLFxuICAgIE9wZW5BcGlPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwYXJzZWQgT3BlbkFQSSBzcGVjaWZpY2F0aW9uXG4gICAqL1xuICByZWFkb25seSBzcGVjOiBhbnk7IC8vIFR5cGUgaXMgT3BlbkFQSVYzLkRvY3VtZW50IC0gaG93ZXZlciBub3QgdHJhbnNwaWxlYWJsZSB2aWEganNpaSwgc28gd2UgdXNlIGFueS5cbiAgLyoqXG4gICAqIFBhdGggdG8gdGhlIEpTT04gb3BlbiBhcGkgc3BlY1xuICAgKi9cbiAgcmVhZG9ubHkgc3BlY1BhdGg6IHN0cmluZztcbn1cblxuLyoqXG4gKiBBIGNvbnN0cnVjdCBmb3IgY3JlYXRpbmcgYW4gYXBpIGdhdGV3YXkgYXBpIGJhc2VkIG9uIHRoZSBkZWZpbml0aW9uIGluIHRoZSBPcGVuQVBJIHNwZWMuXG4gKi9cbmV4cG9ydCBjbGFzcyBPcGVuQXBpR2F0ZXdheUxhbWJkYUFwaSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBhcGk6IFNwZWNSZXN0QXBpO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBwcm9wczogT3BlbkFwaUdhdGV3YXlMYW1iZGFBcGlQcm9wc1xuICApIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qge1xuICAgICAgaW50ZWdyYXRpb25zLFxuICAgICAgc3BlYyxcbiAgICAgIHNwZWNQYXRoLFxuICAgICAgb3BlcmF0aW9uTG9va3VwLFxuICAgICAgZGVmYXVsdEF1dGhvcml6ZXIsXG4gICAgICBjb3JzT3B0aW9ucyxcbiAgICAgIC4uLm9wdGlvbnNcbiAgICB9ID0gcHJvcHM7XG5cbiAgICAvLyBVcGxvYWQgdGhlIHNwZWMgdG8gczMgYXMgYW4gYXNzZXRcbiAgICBjb25zdCBpbnB1dFNwZWNBc3NldCA9IG5ldyBBc3NldCh0aGlzLCBcIklucHV0U3BlY1wiLCB7XG4gICAgICBwYXRoOiBzcGVjUGF0aCxcbiAgICB9KTtcbiAgICAvLyBXZSdsbCBvdXRwdXQgdGhlIHByZXBhcmVkIHNwZWMgaW4gdGhlIHNhbWUgYXNzZXQgYnVja2V0XG4gICAgY29uc3QgcHJlcGFyZWRTcGVjT3V0cHV0S2V5UHJlZml4ID0gYCR7aW5wdXRTcGVjQXNzZXQuczNPYmplY3RLZXl9LXByZXBhcmVkYDtcblxuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG5cbiAgICBjb25zdCBwcmVwYXJlU3BlY0xhbWJkYU5hbWUgPSBgJHtQREtOYWcuZ2V0U3RhY2tQcmVmaXgoc3RhY2spXG4gICAgICAuc3BsaXQoXCIvXCIpXG4gICAgICAuam9pbihcIi1cIil9UHJlcGFyZVNwZWNgO1xuICAgIGNvbnN0IHByZXBhcmVTcGVjUm9sZSA9IG5ldyBSb2xlKHRoaXMsIFwiUHJlcGFyZVNwZWNSb2xlXCIsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoXCJsYW1iZGEuYW1hem9uYXdzLmNvbVwiKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIGxvZ3M6IG5ldyBQb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgXCJsb2dzOkNyZWF0ZUxvZ0dyb3VwXCIsXG4gICAgICAgICAgICAgICAgXCJsb2dzOkNyZWF0ZUxvZ1N0cmVhbVwiLFxuICAgICAgICAgICAgICAgIFwibG9nczpQdXRMb2dFdmVudHNcIixcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgICAgICAgYGFybjphd3M6bG9nczoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvJHtwcmVwYXJlU3BlY0xhbWJkYU5hbWV9YCxcbiAgICAgICAgICAgICAgICBgYXJuOmF3czpsb2dzOiR7c3RhY2sucmVnaW9ufToke3N0YWNrLmFjY291bnR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8ke3ByZXBhcmVTcGVjTGFtYmRhTmFtZX06KmAsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgICAgczM6IG5ldyBQb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbXCJzMzpnZXRPYmplY3RcIl0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGlucHV0U3BlY0Fzc2V0LmJ1Y2tldC5hcm5Gb3JPYmplY3RzKGlucHV0U3BlY0Fzc2V0LnMzT2JqZWN0S2V5KSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbXCJzMzpwdXRPYmplY3RcIl0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIC8vIFRoZSBvdXRwdXQgZmlsZSB3aWxsIGluY2x1ZGUgYSBoYXNoIG9mIHRoZSBwcmVwYXJlZCBzcGVjLCB3aGljaCBpcyBub3Qga25vd24gdW50aWwgZGVwbG95IHRpbWUgc2luY2VcbiAgICAgICAgICAgICAgICAvLyB0b2tlbnMgbXVzdCBiZSByZXNvbHZlZFxuICAgICAgICAgICAgICAgIGlucHV0U3BlY0Fzc2V0LmJ1Y2tldC5hcm5Gb3JPYmplY3RzKFxuICAgICAgICAgICAgICAgICAgYCR7cHJlcGFyZWRTcGVjT3V0cHV0S2V5UHJlZml4fS8qYFxuICAgICAgICAgICAgICAgICksXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICBwcmVwYXJlU3BlY1JvbGUsXG4gICAgICBbXG4gICAgICAgIHtcbiAgICAgICAgICBpZDogXCJBd3NTb2x1dGlvbnMtSUFNNVwiLFxuICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgIFwiQ2xvdWR3YXRjaCByZXNvdXJjZXMgaGF2ZSBiZWVuIHNjb3BlZCBkb3duIHRvIHRoZSBMb2dHcm91cCBsZXZlbCwgaG93ZXZlciAqIGlzIHN0aWxsIG5lZWRlZCBhcyBzdHJlYW0gbmFtZXMgYXJlIGNyZWF0ZWQganVzdCBpbiB0aW1lLlwiLFxuICAgICAgICAgIGFwcGxpZXNUbzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICByZWdleDogYC9eUmVzb3VyY2U6OmFybjphd3M6bG9nczoke1BES05hZy5nZXRTdGFja1JlZ2lvblJlZ2V4KFxuICAgICAgICAgICAgICAgIHN0YWNrXG4gICAgICAgICAgICAgICl9OiR7UERLTmFnLmdldFN0YWNrQWNjb3VudFJlZ2V4KFxuICAgICAgICAgICAgICAgIHN0YWNrXG4gICAgICAgICAgICAgICl9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8ke3ByZXBhcmVTcGVjTGFtYmRhTmFtZX06XFwqL2dgLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWVcbiAgICApO1xuXG4gICAgLy8gQ3JlYXRlIGEgY3VzdG9tIHJlc291cmNlIGZvciBwcmVwYXJpbmcgdGhlIHNwZWMgZm9yIGRlcGxveW1lbnQgKGFkZGluZyBpbnRlZ3JhdGlvbnMsIGF1dGhvcml6ZXJzLCBldGMpXG4gICAgY29uc3QgcHJlcGFyZVNwZWMgPSBuZXcgTGFtYmRhRnVuY3Rpb24odGhpcywgXCJQcmVwYXJlU3BlY1wiLCB7XG4gICAgICBoYW5kbGVyOiBcImluZGV4LmhhbmRsZXJcIixcbiAgICAgIHJ1bnRpbWU6IFJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgICBjb2RlOiBDb2RlLmZyb21Bc3NldChcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgXCIuLi8uLi9saWIvY29uc3RydWN0L3ByZXBhcmUtc3BlYy1ldmVudC1oYW5kbGVyXCIpXG4gICAgICApLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcygzMCksXG4gICAgICByb2xlOiBwcmVwYXJlU3BlY1JvbGUsXG4gICAgICBmdW5jdGlvbk5hbWU6IHByZXBhcmVTcGVjTGFtYmRhTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHByb3ZpZGVyRnVuY3Rpb25OYW1lID0gYCR7cHJlcGFyZVNwZWNMYW1iZGFOYW1lfS1Qcm92aWRlcmA7XG4gICAgY29uc3QgcHJvdmlkZXJSb2xlID0gbmV3IFJvbGUodGhpcywgXCJQcmVwYXJlU3BlY1Byb3ZpZGVyUm9sZVwiLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKFwibGFtYmRhLmFtYXpvbmF3cy5jb21cIiksXG4gICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICBsb2dzOiBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dHcm91cFwiLFxuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dTdHJlYW1cIixcbiAgICAgICAgICAgICAgICBcImxvZ3M6UHV0TG9nRXZlbnRzXCIsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGBhcm46YXdzOmxvZ3M6JHtzdGFjay5yZWdpb259OiR7c3RhY2suYWNjb3VudH06bG9nLWdyb3VwOi9hd3MvbGFtYmRhLyR7cHJvdmlkZXJGdW5jdGlvbk5hbWV9YCxcbiAgICAgICAgICAgICAgICBgYXJuOmF3czpsb2dzOiR7c3RhY2sucmVnaW9ufToke3N0YWNrLmFjY291bnR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8ke3Byb3ZpZGVyRnVuY3Rpb25OYW1lfToqYCxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHByb3ZpZGVyID0gbmV3IFByb3ZpZGVyKHRoaXMsIFwiUHJlcGFyZVNwZWNQcm92aWRlclwiLCB7XG4gICAgICBvbkV2ZW50SGFuZGxlcjogcHJlcGFyZVNwZWMsXG4gICAgICByb2xlOiBwcm92aWRlclJvbGUsXG4gICAgICBwcm92aWRlckZ1bmN0aW9uTmFtZSxcbiAgICB9KTtcblxuICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhcbiAgICAgIHByb3ZpZGVyUm9sZSxcbiAgICAgIFtcbiAgICAgICAge1xuICAgICAgICAgIGlkOiBcIkF3c1NvbHV0aW9ucy1JQU01XCIsXG4gICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgXCJDbG91ZHdhdGNoIHJlc291cmNlcyBoYXZlIGJlZW4gc2NvcGVkIGRvd24gdG8gdGhlIExvZ0dyb3VwIGxldmVsLCBob3dldmVyICogaXMgc3RpbGwgbmVlZGVkIGFzIHN0cmVhbSBuYW1lcyBhcmUgY3JlYXRlZCBqdXN0IGluIHRpbWUuXCIsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdHJ1ZVxuICAgICk7XG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgcHJvdmlkZXIsXG4gICAgICBbXG4gICAgICAgIHtcbiAgICAgICAgICBpZDogXCJBd3NTb2x1dGlvbnMtTDFcIixcbiAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICBcIkxhdGVzdCBydW50aW1lIGNhbm5vdCBiZSBjb25maWd1cmVkLiBDREsgd2lsbCBuZWVkIHRvIHVwZ3JhZGUgdGhlIFByb3ZpZGVyIGNvbnN0cnVjdCBhY2NvcmRpbmdseS5cIixcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB0cnVlXG4gICAgKTtcblxuICAgIGNvbnN0IHByZXBhcmVTcGVjT3B0aW9uczogUHJlcGFyZUFwaVNwZWNPcHRpb25zID0ge1xuICAgICAgaW50ZWdyYXRpb25zOiBPYmplY3QuZnJvbUVudHJpZXMoXG4gICAgICAgIE9iamVjdC5lbnRyaWVzKGludGVncmF0aW9ucykubWFwKChbb3BlcmF0aW9uLCBpbnRlZ3JhdGlvbl0pID0+IFtcbiAgICAgICAgICBvcGVyYXRpb24sXG4gICAgICAgICAge1xuICAgICAgICAgICAgZnVuY3Rpb25JbnZvY2F0aW9uVXJpOiBmdW5jdGlvbkludm9jYXRpb25VcmkoXG4gICAgICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgICAgIGludGVncmF0aW9uLmZ1bmN0aW9uXG4gICAgICAgICAgICApLFxuICAgICAgICAgICAgbWV0aG9kQXV0aG9yaXplcjogc2VyaWFsaXplQXNBdXRob3JpemVyUmVmZXJlbmNlKFxuICAgICAgICAgICAgICBpbnRlZ3JhdGlvbi5hdXRob3JpemVyIHx8IGRlZmF1bHRBdXRob3JpemVyIHx8IEF1dGhvcml6ZXJzLm5vbmUoKVxuICAgICAgICAgICAgKSxcbiAgICAgICAgICB9LFxuICAgICAgICBdKVxuICAgICAgKSxcbiAgICAgIHNlY3VyaXR5U2NoZW1lczogcHJlcGFyZVNlY3VyaXR5U2NoZW1lcyhcbiAgICAgICAgdGhpcyxcbiAgICAgICAgaW50ZWdyYXRpb25zLFxuICAgICAgICBkZWZhdWx0QXV0aG9yaXplclxuICAgICAgKSxcbiAgICAgIGNvcnNPcHRpb25zOiBjb3JzT3B0aW9ucyAmJiB7XG4gICAgICAgIGFsbG93SGVhZGVyczogY29yc09wdGlvbnMuYWxsb3dIZWFkZXJzIHx8IENvcnMuREVGQVVMVF9IRUFERVJTLFxuICAgICAgICBhbGxvd01ldGhvZHM6IGNvcnNPcHRpb25zLmFsbG93TWV0aG9kcyB8fCBDb3JzLkFMTF9NRVRIT0RTLFxuICAgICAgICBhbGxvd09yaWdpbnM6IGNvcnNPcHRpb25zLmFsbG93T3JpZ2lucyxcbiAgICAgICAgc3RhdHVzQ29kZTogY29yc09wdGlvbnMuc3RhdHVzQ29kZSB8fCAyMDQsXG4gICAgICB9LFxuICAgICAgb3BlcmF0aW9uTG9va3VwLFxuICAgIH07XG5cbiAgICAvLyBTcGVjIHByZXBhcmF0aW9uIHdpbGwgaGFwcGVuIGluIGEgY3VzdG9tIHJlc291cmNlIGxhbWJkYSBzbyB0aGF0IHJlZmVyZW5jZXMgdG8gbGFtYmRhIGludGVncmF0aW9ucyBldGMgY2FuIGJlXG4gICAgLy8gcmVzb2x2ZWQuIEhvd2V2ZXIsIHdlIGFsc28gcHJlcGFyZSBpbmxpbmUgdG8gcGVyZm9ybSBzb21lIGFkZGl0aW9uYWwgdmFsaWRhdGlvbiBhdCBzeW50aCB0aW1lLlxuICAgIGNvbnN0IHByZXBhcmVkU3BlYyA9IHByZXBhcmVBcGlTcGVjKHNwZWMsIHByZXBhcmVTcGVjT3B0aW9ucyk7XG5cbiAgICBjb25zdCBwcmVwYXJlQXBpU3BlY0N1c3RvbVJlc291cmNlUHJvcGVydGllczogUHJlcGFyZUFwaVNwZWNDdXN0b21SZXNvdXJjZVByb3BlcnRpZXMgPVxuICAgICAge1xuICAgICAgICBpbnB1dFNwZWNMb2NhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldDogaW5wdXRTcGVjQXNzZXQuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgICAga2V5OiBpbnB1dFNwZWNBc3NldC5zM09iamVjdEtleSxcbiAgICAgICAgfSxcbiAgICAgICAgb3V0cHV0U3BlY0xvY2F0aW9uOiB7XG4gICAgICAgICAgYnVja2V0OiBpbnB1dFNwZWNBc3NldC5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICBrZXk6IHByZXBhcmVkU3BlY091dHB1dEtleVByZWZpeCxcbiAgICAgICAgfSxcbiAgICAgICAgLi4ucHJlcGFyZVNwZWNPcHRpb25zLFxuICAgICAgfTtcblxuICAgIGNvbnN0IHByZXBhcmVTcGVjQ3VzdG9tUmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UoXG4gICAgICB0aGlzLFxuICAgICAgXCJQcmVwYXJlU3BlY0N1c3RvbVJlc291cmNlXCIsXG4gICAgICB7XG4gICAgICAgIHNlcnZpY2VUb2tlbjogcHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgICAgICBwcm9wZXJ0aWVzOiBwcmVwYXJlQXBpU3BlY0N1c3RvbVJlc291cmNlUHJvcGVydGllcyxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBhcGkgZ2F0ZXdheSByZXNvdXJjZXMgZnJvbSB0aGUgc3BlYywgYXVnbWVudGluZyB0aGUgc3BlYyB3aXRoIHRoZSBwcm9wZXJ0aWVzIHNwZWNpZmljIHRvIGFwaSBnYXRld2F5XG4gICAgLy8gc3VjaCBhcyBpbnRlZ3JhdGlvbnMgb3IgYXV0aCB0eXBlc1xuICAgIHRoaXMuYXBpID0gbmV3IFNwZWNSZXN0QXBpKHRoaXMsIGlkLCB7XG4gICAgICBhcGlEZWZpbml0aW9uOiBBcGlEZWZpbml0aW9uLmZyb21CdWNrZXQoXG4gICAgICAgIGlucHV0U3BlY0Fzc2V0LmJ1Y2tldCxcbiAgICAgICAgcHJlcGFyZVNwZWNDdXN0b21SZXNvdXJjZS5nZXRBdHRTdHJpbmcoXCJvdXRwdXRTcGVjS2V5XCIpXG4gICAgICApLFxuICAgICAgZGVwbG95T3B0aW9uczoge1xuICAgICAgICBhY2Nlc3NMb2dEZXN0aW5hdGlvbjogbmV3IExvZ0dyb3VwTG9nRGVzdGluYXRpb24oXG4gICAgICAgICAgbmV3IExvZ0dyb3VwKHNjb3BlLCBgQWNjZXNzTG9nc2ApXG4gICAgICAgICksXG4gICAgICAgIGFjY2Vzc0xvZ0Zvcm1hdDogQWNjZXNzTG9nRm9ybWF0LmNsZigpLFxuICAgICAgICBsb2dnaW5nTGV2ZWw6IE1ldGhvZExvZ2dpbmdMZXZlbC5JTkZPLFxuICAgICAgfSxcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgfSk7XG5cbiAgICB0aGlzLmFwaS5ub2RlLmFkZERlcGVuZGVuY3kocHJlcGFyZVNwZWNDdXN0b21SZXNvdXJjZSk7XG5cbiAgICAvLyBXaGlsZSB0aGUgYXBpIHdpbGwgYmUgdXBkYXRlZCB3aGVuIHRoZSBvdXRwdXQgcGF0aCBmcm9tIHRoZSBjdXN0b20gcmVzb3VyY2UgY2hhbmdlcywgQ0RLIHN0aWxsIG5lZWRzIHRvIGtub3cgd2hlblxuICAgIC8vIHRvIHJlZGVwbG95IHRoZSBhcGkuIFRoaXMgaXMgYWNoaWV2ZWQgYnkgaW5jbHVkaW5nIGEgaGFzaCBvZiB0aGUgc3BlYyBpbiB0aGUgbG9naWNhbCBpZCAoaW50ZXJuYWxpc2VkIGluIHRoZVxuICAgIC8vIGFkZFRvTG9naWNhbElkIG1ldGhvZCBzaW5jZSB0aGlzIGlzIGhvdyBjaGFuZ2VzIG9mIGluZGl2aWR1YWwgcmVzb3VyY2VzL21ldGhvZHMgZXRjIHRyaWdnZXIgcmVkZXBsb3ltZW50cyBpbiBDREspXG4gICAgdGhpcy5hcGkubGF0ZXN0RGVwbG95bWVudD8uYWRkVG9Mb2dpY2FsSWQocHJlcGFyZWRTcGVjKTtcblxuICAgIC8vIEdyYW50IEFQSSBHYXRld2F5IHBlcm1pc3Npb24gdG8gaW52b2tlIGVhY2ggbGFtYmRhIHdoaWNoIGltcGxlbWVudHMgYW4gaW50ZWdyYXRpb24gb3IgY3VzdG9tIGF1dGhvcml6ZXJcbiAgICBnZXRMYWJlbGxlZEZ1bmN0aW9ucyhwcm9wcykuZm9yRWFjaCgoeyBsYWJlbCwgZnVuY3Rpb246IGxhbWJkYSB9KSA9PiB7XG4gICAgICBuZXcgQ2ZuUGVybWlzc2lvbih0aGlzLCBgTGFtYmRhUGVybWlzc2lvbi0ke2xhYmVsfWAsIHtcbiAgICAgICAgYWN0aW9uOiBcImxhbWJkYTpJbnZva2VGdW5jdGlvblwiLFxuICAgICAgICBwcmluY2lwYWw6IFwiYXBpZ2F0ZXdheS5hbWF6b25hd3MuY29tXCIsXG4gICAgICAgIGZ1bmN0aW9uTmFtZTogbGFtYmRhLmZ1bmN0aW9uQXJuLFxuICAgICAgICBzb3VyY2VBcm46IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgc2VydmljZTogXCJleGVjdXRlLWFwaVwiLFxuICAgICAgICAgIHJlc291cmNlOiB0aGlzLmFwaS5yZXN0QXBpSWQsXG4gICAgICAgICAgcmVzb3VyY2VOYW1lOiBcIiovKlwiLFxuICAgICAgICB9KSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgdGhpcyxcbiAgICAgIFtcbiAgICAgICAge1xuICAgICAgICAgIGlkOiBcIkF3c1NvbHV0aW9ucy1JQU00XCIsXG4gICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgXCJDbG91ZHdhdGNoIFJvbGUgcmVxdWlyZXMgYWNjZXNzIHRvIGNyZWF0ZS9yZWFkIGdyb3VwcyBhdCB0aGUgcm9vdCBsZXZlbC5cIixcbiAgICAgICAgICBhcHBsaWVzVG86IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgcmVnZXg6IGAvXlBvbGljeTo6YXJuOiR7UERLTmFnLmdldFN0YWNrUGFydGl0aW9uUmVnZXgoXG4gICAgICAgICAgICAgICAgc3RhY2tcbiAgICAgICAgICAgICAgKX06aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BbWF6b25BUElHYXRld2F5UHVzaFRvQ2xvdWRXYXRjaExvZ3MkL2dgLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWVcbiAgICApO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgdGhpcyxcbiAgICAgIFtcbiAgICAgICAge1xuICAgICAgICAgIGlkOiBcIkF3c1NvbHV0aW9ucy1BUElHMlwiLFxuICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgIFwiVGhpcyBjb25zdHJ1Y3QgaW1wbGVtZW50cyBmaW5lIGdyYWluZWQgdmFsaWRhdGlvbiB2aWEgT3BlbkFwaS5cIixcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB0cnVlXG4gICAgKTtcbiAgfVxufVxuIl19