"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuildProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const sample_executable_1 = require("./components/sample-executable");
const smithy_build_gradle_file_1 = require("./components/smithy-build-gradle-file");
const smithy_generated_output_1 = require("./components/smithy-generated-output");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyBuildProject extends projen_1.Project {
    constructor(options) {
        super(options);
        // Store whether we've synthesized the project
        this.synthed = false;
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        const samplePath = path.join(__dirname, "..", "..", "..", "samples", "smithy");
        // Add gradle wrapper files and executables
        [
            "gradle/wrapper/gradle-wrapper.jar",
            "gradle/wrapper/gradle-wrapper.properties",
        ].forEach((file) => {
            new projen_1.SampleFile(this, file, {
                sourcePath: path.join(samplePath, file),
            });
        });
        ["gradlew", "gradlew.bat"].forEach((executable) => {
            new sample_executable_1.SampleExecutable(this, executable, {
                sourcePath: path.join(samplePath, executable),
            });
        });
        // Add settings.gradle
        new projen_1.SampleFile(this, "settings.gradle", {
            contents: `rootProject.name = '${this.name.replace(/[\/\\:<>"?\*|]/g, "-")}'`,
        });
        const modelDir = "src/main/smithy";
        // Always add the following required dependencies
        const requiredDependencies = [
            "software.amazon.smithy:smithy-cli",
            "software.amazon.smithy:smithy-model",
            "software.amazon.smithy:smithy-openapi",
            "software.amazon.smithy:smithy-aws-traits",
        ];
        const requiredSmithyDependencyVersion = "1.28.0";
        // Ensure dependencies always include the required dependencies, allowing users to customise the version
        const userSpecifiedDependencies = options.smithyBuildOptions?.maven?.dependencies ?? [];
        const userSpecifiedDependencySet = new Set(userSpecifiedDependencies.map((dep) => dep.split(":").slice(0, -1).join(":")));
        const dependencies = [
            ...requiredDependencies
                .filter((requiredDep) => !userSpecifiedDependencySet.has(requiredDep))
                .map((dep) => `${dep}:${requiredSmithyDependencyVersion}`),
            ...userSpecifiedDependencies,
        ];
        // Add build.gradle
        new smithy_build_gradle_file_1.SmithyBuildGradleFile(this, {
            modelDir,
            dependencies,
            repositoryUrls: options.smithyBuildOptions?.maven?.repositoryUrls,
        });
        const { namespace: serviceNamespace, serviceName } = options.serviceName;
        // Create the default smithy model
        new projen_1.SampleFile(this, path.join(modelDir, "main.smithy"), {
            contents: `$version: "2"
namespace ${serviceNamespace}

use aws.protocols#restJson1

/// A sample smithy api
@restJson1
service ${serviceName} {
    version: "1.0"
    operations: [SayHello]
}

@readonly
@http(method: "GET", uri: "/hello")
operation SayHello {
    input: SayHelloInput
    output: SayHelloOutput
    errors: [ApiError]
}

string Name
string Message

@input
structure SayHelloInput {
    @httpQuery("name")
    @required
    name: Name
}

@output
structure SayHelloOutput {
    @required
    message: Message
}

@error("client")
structure ApiError {
    @required
    errorMessage: Message
}
`,
        });
        // Create the smithy build json file
        new smithy_build_1.SmithyBuild(this, {
            version: "2.0",
            ...options.smithyBuildOptions,
            projections: {
                ...options.smithyBuildOptions?.projections,
                openapi: {
                    plugins: {
                        openapi: {
                            service: `${serviceNamespace}#${serviceName}`,
                            // By default, preserve tags in the generated spec, but allow users to explicitly overwrite this
                            tags: true,
                            ...options.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
            maven: {
                dependencies,
                repositories: (options.smithyBuildOptions?.maven?.repositoryUrls ?? [
                    "https://repo.maven.apache.org/maven2/",
                    "file://~/.m2/repository",
                ]).map((url) => ({ url })),
            },
        });
        // SmithyBuild component above always writes to smithy-build.json
        this.smithyBuildConfigPath = path.join(this.outdir, "smithy-build.json");
        this.smithyBuildOutputPath = path.join(this.outdir, options.buildOutputDir);
        new smithy_generated_output_1.SmithyGeneratedOutput(this, {
            modelPath: path.join(this.outdir, modelDir),
            gradleProjectPath: this.outdir,
            smithyBuildConfigPath: this.smithyBuildConfigPath,
            outputPath: this.smithyBuildOutputPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.SmithyBuildProject = SmithyBuildProject;
//# sourceMappingURL=data:application/json;base64,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