"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const ec2 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func).toBeDefined();
    // Assertion 2
    const secret = pattern.secret;
    expect(secret).toBeDefined();
});
// --------------------------------------------------------------
// Test deployment w/ existing secret
// --------------------------------------------------------------
test('Test deployment w/ existing secret', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const existingSecret = new aws_secretsmanager_1.Secret(stack, 'secret', {});
    const pattern = new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        existingSecretObj: existingSecret
    });
    // Assertion 1
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
    });
    // Assertion 2
    expect(pattern.secret).toBe(existingSecret);
});
// --------------------------------------------------------------
// Test deployment w/ existing function
// --------------------------------------------------------------
test('Test deployment w/ existing function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const existingFunction = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    const pattern = new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        existingLambdaObj: existingFunction,
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
    });
    // Assertion 1
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
    });
    // Assertion 2
    expect(pattern.lambdaFunction).toBe(existingFunction);
});
// --------------------------------------------------------------
// Test minimal deployment with write access to Secret
// --------------------------------------------------------------
test('Test minimal deployment write access to Secret', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
        grantWriteAccess: 'ReadWrite'
    });
    // Assertion 1
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
    });
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosecretsmanagerstackReplaceDefaultSecurityGroupsecuritygroupED420155",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosecretsmanagerstackReplaceDefaultSecurityGroupsecuritygroupED420155",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
        existingVpc: testVpc,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosecretsmanagerstackReplaceDefaultSecurityGroupsecuritygroupED420155",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC and existing Lambda function not in a VPC
//
// buildLambdaFunction should throw an error if the Lambda function is not
// attached to a VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
            secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test bad call with existingVpc and deployVpc
// --------------------------------------------------------------
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
            }
        },
        secretProps: { removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY },
        secretEnvironmentVariableName: 'CUSTOM_SECRET_NAME'
    });
    // Assertion
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_SECRET_NAME: {
                    Ref: 'lambdatosecretsmanagerstacksecretBA684E34'
                }
            }
        }
    });
});
// --------------------------------------------------------------
// Test overriding secretProps to pass a customer provided CMK
// --------------------------------------------------------------
test('Test overriding secretProps to pass a customer provided CMK', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const encryptionKey = defaults.buildEncryptionKey(stack, {
        description: 'secret-key'
    });
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
            }
        },
        secretProps: {
            encryptionKey
        }
    });
    // Assertion 1
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                SECRET_ARN: {
                    Ref: 'lambdatosecretsmanagerstacksecretBA684E34'
                }
            }
        }
    });
    // Assertion 2
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
        KmsKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    // Assertion 3
    expect(stack).toHaveResource('AWS::KMS::Key', {
        Description: "secret-key",
        EnableKeyRotation: true
    });
});
//# sourceMappingURL=data:application/json;base64,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