"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": [
        "export * from './api';",
        ...(options.sampleCode !== false ? ["export * from './sample-api';"] : []),
    ].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.ts": `import { OpenApiGatewayLambdaApi, OpenApiGatewayLambdaApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import * as path from "path";
import spec from "../${options.specDir}/${options.parsedSpecFileName}";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayLambdaApiProps, "spec" | "specPath" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
export class Api extends OpenApiGatewayLambdaApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      specPath: path.resolve(__dirname, "../${options.specDir}/${options.parsedSpecFileName}"),
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Authorizers } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      defaultAuthorizer: Authorizers.iam(),
      integrations: {
        sayHello: {
          function: new NodejsFunction(scope, "say-hello"),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "sample-api.say-hello.ts": `import { sayHelloHandler } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler(async (input) => {
  return {
    statusCode: 200,
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.apiSrcDir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        function: new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        }),
      }])) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,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