"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 *
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 *
 * @stability stable
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    CodeCommitTrigger["NONE"] = "None";
    CodeCommitTrigger["POLL"] = "Poll";
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 *
 * @stability stable
 */
class CodeCommitSourceAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        var _b;
        const branch = (_b = props.branch) !== null && _b !== void 0 ? _b : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        if (props.codeBuildCloneOutput === true) {
            props.output.setMetadata(CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY, props.repository.repositoryArn);
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     *
     * @stability stable
     */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    /**
     * This is a renamed version of the {@link IAction.bind} method.
     *
     * @stability stable
     */
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline, {
                    eventRole: this.props.eventRole,
                }),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // when this action is cross-account,
        // the Role needs the s3:PutObjectAcl permission for some not yet fully understood reason
        if (core_1.Token.compareStrings(this.props.repository.env.account, core_1.Stack.of(stage.pipeline).account) === core_1.TokenComparison.DIFFERENT) {
            options.bucket.grantPutAcl(options.role);
        }
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
                ...(this.props.codeBuildCloneOutput === true ? ['codecommit:GetRepository'] : []),
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
                OutputArtifactFormat: this.props.codeBuildCloneOutput === true
                    ? 'CODEBUILD_CLONE_REF'
                    : undefined,
            },
        };
    }
    generateEventId(stage) {
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node);
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : `-${this.branch}-`;
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
_a = JSII_RTTI_SYMBOL_1;
CodeCommitSourceAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CodeCommitSourceAction", version: "1.116.0" };
/**
 * The name of the property that holds the ARN of the CodeCommit Repository
 * inside of the CodePipeline Artifact's metadata.
 *
 * @internal
 */
CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY = 'CodeCommitCloneRepositoryArn';
//# sourceMappingURL=data:application/json;base64,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