"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...options?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
        ].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientLanguages, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in __init__.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
        // Generate documentation if requested
        if ((options.documentationFormats ?? []).length > 0) {
            new docs_project_1.DocsProject({
                parent: this,
                outdir: path.join(this.generatedCodeDir, "documentation"),
                name: "docs",
                formats: [...new Set(options.documentationFormats)],
                specPath: spec.parsedSpecPath,
            });
        }
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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