from __future__ import annotations

from enum import Enum
from pathlib import Path

import typer

from module_qc_analysis_tools import data

CONTEXT_SETTINGS = {"help_option_names": ["-h", "--help"]}


class FitMethod(str, Enum):
    root = "root"
    numpy = "numpy"


class LogLevel(str, Enum):
    debug = "DEBUG"
    info = "INFO"
    warning = "WARNING"
    error = "ERROR"


OPTIONS = {}

OPTIONS["input_meas"]: Path = typer.Option(
    ...,
    "-i",
    "--input-meas",
    help="path to the input measurement file or directory containing input measurement files.",
    exists=True,
    file_okay=True,
    readable=True,
    resolve_path=True,
)
OPTIONS["output_dir"]: Path = typer.Option(
    "outputs",
    "-o",
    "--output-dir",
    help="output directory",
    exists=False,
    writable=True,
)
OPTIONS["qc_criteria"]: Path = typer.Option(
    str(data / "analysis_cuts.json"),
    "-q",
    "--qc-criteria",
    help="path to json file with QC selection criteria (default: $(module-qc-analysis-tools --prefix)/analysis_cuts.json)",
    exists=True,
    file_okay=True,
    readable=True,
    resolve_path=True,
)
OPTIONS["layer"]: str = typer.Option(
    "Unknown",
    "-l",
    "--layer",
    help="Layer of module, used for applying correct QC criteria settings. Options: L0, L1, L2 (default)",
)
OPTIONS["permodule"]: bool = typer.Option(
    False, help="Store results in one file per module (default: one file per chip)"
)
OPTIONS["fit_method"]: FitMethod = typer.Option(
    FitMethod.numpy, "-f", "--fit-method", help="fitting method"
)
OPTIONS["nchips"]: int = typer.Option(
    4,
    "-n",
    "--nChips",
    help="Number of chips powered in parallel (e.g. 4 for a quad module, 3 for a triplet, 1 for an SCC.)",
)
OPTIONS["verbosity"]: LogLevel = typer.Option(
    LogLevel.info,
    "-v",
    "--verbosity",
    help="Log level [options: DEBUG, INFO (default) WARNING, ERROR]",
)
OPTIONS["submit"]: bool = typer.Option(
    False,
    help="Submit QC results to: https://docs.google.com/spreadsheets/d/1pw_07F94fg2GJQr8wlvhaRUV63uhsAuBt_S1FEFBzBU/view",
)
OPTIONS["site"]: str = typer.Option(
    "",
    "--site",
    help="Your testing site. Required when submitting results. Please use same short-hand as on production DB, i.e. LBNL_PIXEL_MODULES for LBNL, IRFU for Paris-Saclay, ...",
)
OPTIONS["lp_enable"]: bool = typer.Option(False, help="low power mode")

OPTIONS["input_file"]: Path = typer.Option(
    ...,
    "-i",
    "--input-file",
    help="analysis output file",
    exists=True,
    file_okay=True,
    readable=True,
    resolve_path=True,
)

OPTIONS["config_file"]: Path = typer.Option(
    ...,
    "-c",
    "--config-file",
    help="the config file to be modified",
    exists=True,
    file_okay=True,
    readable=True,
    writable=True,
    resolve_path=True,
)

OPTIONS["input_dir"]: Path = typer.Option(
    ...,
    "-i",
    "--input-dir",
    help="Analysis output directory",
    exists=True,
    dir_okay=True,
    readable=True,
    resolve_path=True,
)
OPTIONS["config_dir"]: Path = typer.Option(
    ...,
    "-c",
    "--config-dir",
    help="Path to the module configuration directory to be modified",
    exists=True,
    dir_okay=True,
    readable=True,
    writable=True,
    resolve_path=True,
)
OPTIONS["config_type"]: str = typer.Option(
    "",
    "-t",
    "--config-type",
    help="The config type to be modified. E.g. E.g. L2_warm/l2_cold.",
)
OPTIONS["override"]: bool = typer.Option(
    False,
    help="Update chip configuration even if the chip failed QC",
)
