import torch
from torch.optim import AdamW


class ElectraAdamW(AdamW):
    r"""Implements AdamW algorithm.

    The original Adam algorithm was proposed in `Adam: A Method for Stochastic Optimization`_.
    The AdamW variant was proposed in `Decoupled Weight Decay Regularization`_. This version is
    inspired to the version of AdamW proposed in `ELECTRA: PRE-TRAINING TEXT ENCODERS
    AS DISCRIMINATORS RATHER THAN GENERATORS`. This code is the porting of the original
    tensorflow code available `here <https://github.com/google-research/electra/blob/master/model/optimization.py>`_.

    Arguments:
        params (iterable): iterable of parameters to optimize or dicts defining
            parameter groups
        lr (float, optional): learning rate (default: 1e-3)
        betas (Tuple[float, float], optional): coefficients used for computing
            running averages of gradient and its square (default: (0.9, 0.999))
        eps (float, optional): term added to the denominator to improve
            numerical stability (default: 1e-8)
        weight_decay (float, optional): weight decay coefficient (default: 1e-2)
        amsgrad (boolean, optional): whether to use the AMSGrad variant of this
            algorithm from the paper `On the Convergence of Adam and Beyond`_
            (default: False)

    .. _Adam\: A Method for Stochastic Optimization:
        https://arxiv.org/abs/1412.6980
    .. _Decoupled Weight Decay Regularization:
        https://arxiv.org/abs/1711.05101
    .. _On the Convergence of Adam and Beyond:
        https://openreview.net/forum?id=ryQu7f-RZ
    """

    def __init__(self, params, lr=1e-3, betas=(0.9, 0.999), eps=1e-6, weight_decay=1e-2, amsgrad=False):
        super(ElectraAdamW,
              self).__init__(params, lr=lr, betas=betas, eps=eps, weight_decay=weight_decay, amsgrad=amsgrad)

    @torch.no_grad()
    def step(self, closure=None):
        """Performs a single optimization step.

        Arguments:
            closure (callable, optional): A closure that reevaluates the model
                and returns the loss.
        """
        loss = None
        if closure is not None:
            with torch.enable_grad():
                loss = closure()

        for group in self.param_groups:
            for p in group['params']:
                if p.grad is None:
                    continue

                # Perform optimization step
                grad = p.grad
                state = self.state[p]

                # State initialization
                if len(state) == 0:
                    # Exponential moving average of gradient values
                    state['exp_avg'] = torch.zeros_like(p, memory_format=torch.preserve_format)
                    # Exponential moving average of squared gradient values
                    state['exp_avg_sq'] = torch.zeros_like(p, memory_format=torch.preserve_format)

                beta1, beta2 = group['betas']
                exp_avg, exp_avg_sq = state['exp_avg'], state['exp_avg_sq']

                next_exp_avg = beta1 * exp_avg + (1 - beta1) * grad
                next_exp_avg_sq = beta2 * exp_avg_sq + (1 - beta2) * torch.square(grad)

                # Decay the first and second moment running average coefficient
                update = next_exp_avg / (torch.sqrt(next_exp_avg_sq) + group['eps'])

                # Perform stepweight decay
                update += p * group['weight_decay']

                # Update parameters
                p.sub_(group['lr'] * update)

                # Update internal state
                state['exp_avg'], state['exp_avg_sq'] = next_exp_avg, next_exp_avg_sq

        return loss
