"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.getLabelledFunctions = void 0;
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const openapi_types_1 = require("openapi-types");
const authorizers_1 = require("../authorizers");
const predicates_1 = require("../authorizers/predicates");
const api_gateway_auth_1 = require("./api-gateway-auth");
const utils_1 = require("./utils");
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (scope, path, method, { integrations, defaultAuthorizer, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { authorizer, function: lambdaFunction } = integrations[operationName];
    // Prefer the integration level authorizer, otherwise use the default, or none if unspecified
    const methodAuthorizer = authorizer ?? defaultAuthorizer ?? authorizers_1.Authorizers.none();
    // Generate the lambda invocation arn as the uri for the API Gateway integration
    const uri = utils_1.functionInvocationUri(scope, lambdaFunction);
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri,
            passthroughBehavior: aws_apigateway_1.PassthroughBehavior.WHEN_NO_MATCH,
        },
        ...api_gateway_auth_1.applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${(corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS).join(",")}'`,
    "Access-Control-Allow-Methods": `'${(corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS).join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (openapi_types_1.OpenAPIV3.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode || 204;
    return {
        [openapi_types_1.OpenAPIV3.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (scope, path, pathItem, options, getOperationName) => {
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(openapi_types_1.OpenAPIV3.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(scope, path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return all lambda functions used as authorizers, labelled by authorizer id
 */
const getLabelledAuthorizerFunctions = (options) => api_gateway_auth_1.getAllAuthorizers(options)
    .filter((authorizer) => predicates_1.isCustomAuthorizer(authorizer))
    .map((authorizer) => ({
    label: authorizer.authorizerId,
    function: authorizer.function,
}));
/**
 * Return all lambda functions used in integrations, labelled by operation
 */
const getLabelledIntegrationFunctions = (options) => Object.entries(options.integrations).map(([operationId, integration]) => ({
    label: operationId,
    function: integration.function,
}));
/**
 * Return all lambda functions that may be invoked by api gateway
 */
exports.getLabelledFunctions = (options) => getLabelledAuthorizerFunctions(options).concat(getLabelledIntegrationFunctions(options));
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (scope, spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        utils_1.concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[utils_1.concatMethodAndPath(methodAndPath)];
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            [aws_apigateway_1.ResponseType.BAD_REQUEST_BODY.responseType]: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(scope, path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(scope, options),
        },
    };
};
//# sourceMappingURL=data:application/json;base64,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