"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnvironmentVariablesService = void 0;
const commons_1 = require("@aws-lambda-powertools/commons");
/**
 * Class EnvironmentVariablesService
 *
 * This class is used to return environment variables that are available in the runtime of
 * the current Lambda invocation.
 * These variables can be a mix of runtime environment variables set by AWS and
 * variables that can be set by the developer additionally.
 *
 * @class
 * @extends {CommonEnvironmentVariablesService}
 * @implements {ConfigServiceInterface}
 * @see https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html#configuration-envvars-runtime
 * @see https://awslabs.github.io/aws-lambda-powertools-typescript/latest/#environment-variables
 */
class EnvironmentVariablesService extends commons_1.EnvironmentVariablesService {
    constructor() {
        super(...arguments);
        // Reserved environment variables
        this.awsRegionVariable = 'AWS_REGION';
        this.currentEnvironmentVariable = 'ENVIRONMENT';
        this.devModeVariable = 'POWERTOOLS_DEV';
        this.functionNameVariable = 'AWS_LAMBDA_FUNCTION_NAME';
        this.functionVersionVariable = 'AWS_LAMBDA_FUNCTION_VERSION';
        this.logEventVariable = 'POWERTOOLS_LOGGER_LOG_EVENT';
        this.logLevelVariable = 'LOG_LEVEL';
        this.memoryLimitInMBVariable = 'AWS_LAMBDA_FUNCTION_MEMORY_SIZE';
        this.sampleRateValueVariable = 'POWERTOOLS_LOGGER_SAMPLE_RATE';
    }
    /**
     * It returns the value of the AWS_REGION environment variable.
     *
     * @returns {string}
     */
    getAwsRegion() {
        return this.get(this.awsRegionVariable);
    }
    /**
     * It returns the value of the ENVIRONMENT environment variable.
     *
     * @returns {string}
     */
    getCurrentEnvironment() {
        return this.get(this.currentEnvironmentVariable);
    }
    /**
     * It returns the value of the AWS_LAMBDA_FUNCTION_MEMORY_SIZE environment variable.
     *
     * @returns {string}
     */
    getFunctionMemory() {
        const value = this.get(this.memoryLimitInMBVariable);
        return Number(value);
    }
    /**
     * It returns the value of the AWS_LAMBDA_FUNCTION_NAME environment variable.
     *
     * @returns {string}
     */
    getFunctionName() {
        return this.get(this.functionNameVariable);
    }
    /**
     * It returns the value of the AWS_LAMBDA_FUNCTION_VERSION environment variable.
     *
     * @returns {string}
     */
    getFunctionVersion() {
        return this.get(this.functionVersionVariable);
    }
    /**
     * It returns the value of the POWERTOOLS_LOGGER_LOG_EVENT environment variable.
     *
     * @returns {boolean}
     */
    getLogEvent() {
        const value = this.get(this.logEventVariable);
        return this.isValueTrue(value);
    }
    /**
     * It returns the value of the LOG_LEVEL environment variable.
     *
     * @returns {string}
     */
    getLogLevel() {
        return this.get(this.logLevelVariable);
    }
    /**
     * It returns the value of the POWERTOOLS_LOGGER_SAMPLE_RATE environment variable.
     *
     * @returns {string|undefined}
     */
    getSampleRateValue() {
        const value = this.get(this.sampleRateValueVariable);
        return (value && value.length > 0) ? Number(value) : undefined;
    }
    /**
     * It returns true if the POWERTOOLS_DEV environment variable is set to truthy value.
     *
     * @returns {boolean}
     */
    isDevMode() {
        const value = this.get(this.devModeVariable);
        return this.isValueTrue(value);
    }
    /**
     * It returns true if the string value represents a boolean true value.
     *
     * @param {string} value
     * @returns boolean
     */
    isValueTrue(value) {
        const truthyValues = ['1', 'y', 'yes', 't', 'true', 'on'];
        return truthyValues.includes(value.toLowerCase());
    }
}
exports.EnvironmentVariablesService = EnvironmentVariablesService;
//# sourceMappingURL=data:application/json;base64,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