"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Utility = void 0;
/**
 * ## Intro
 * Utility is a base class that other Powertools utilites can extend to inherit shared logic.
 *
 *
 * ## Key features
 *   * Cold Start heuristic to determine if the current
 *
 * ## Usage
 *
 * ### Cold Start
 *
 * Cold start is a term commonly used to describe the `Init` phase of a Lambda function. In this phase, Lambda creates or unfreezes an execution environment with the configured resources, downloads the code for the function and all layers, initializes any extensions, initializes the runtime, and then runs the function’s initialization code (the code outside the main handler). The Init phase happens either during the first invocation, or in advance of function invocations if you have enabled provisioned concurrency.
 *
 * To learn more about the Lambda execution environment lifecycle, see the [Execution environment section](https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html) of the AWS Lambda documentation.
 *
 * As a Powertools user you probably won't be using this class directly, in fact if you use other Powertools utilities the cold start heuristic found here is already used to:
 * * Add a `coldStart` key to the structured logs when injecting context information in `Logger`
 * * Emit a metric during a cold start function invocation in `Metrics`
 * * Annotate the invocation segment with a `coldStart` key in `Tracer`
 *
 * If you want to use this logic in your own utilities, `Utility` provides two methods:
 *
 * #### `getColdStart()`
 *
 * Since the `Utility` class is instantiated outside of the Lambda handler it will persist across invocations of the same execution environment. This means that if you call `getColdStart()` multiple times, it will return `true` during the first invocation, and `false` afterwards.
 *
 * @example
 * ```typescript
 * import { Utility } from '@aws-lambda-powertools/commons';
 *
 * const utility = new Utility();
 *
 * export const handler = async (_event: any, _context: any) => {
 *   utility.getColdStart();
 * };
 * ```
 *
 * #### `isColdStart()`
 *
 * This method is an alias of `getColdStart()` and is exposed for convenience and better readability in certain usages.
 *
 * @example
 * ```typescript
 * import { Utility } from '@aws-lambda-powertools/commons';
 *
 * const utility = new Utility();
 *
 * export const handler = async (_event: any, _context: any) => {
 *   if (utility.isColdStart()) {
 *     // do something, this block is only executed on the first invocation of the function
 *   } else {
 *     // do something else, this block gets executed on all subsequent invocations
 *   }
 * };
 * ```
 */
class Utility {
    constructor() {
        this.coldStart = true;
        this.defaultServiceName = 'service_undefined';
    }
    getColdStart() {
        if (this.coldStart) {
            this.coldStart = false;
            return true;
        }
        return false;
    }
    isColdStart() {
        return this.getColdStart();
    }
    getDefaultServiceName() {
        return this.defaultServiceName;
    }
    /**
   * Validate that the service name provided is valid.
   * Used internally during initialization.
   *
   * @param serviceName - Service name to validate
   */
    isValidServiceName(serviceName) {
        return typeof serviceName === 'string' && serviceName.trim().length > 0;
    }
}
exports.Utility = Utility;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiVXRpbGl0eS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9VdGlsaXR5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXdERztBQUNILE1BQWEsT0FBTztJQUFwQjtRQUNVLGNBQVMsR0FBWSxJQUFJLENBQUM7UUFDakIsdUJBQWtCLEdBQVcsbUJBQW1CLENBQUM7SUE2QnBFLENBQUM7SUEzQlEsWUFBWTtRQUNqQixJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUM7WUFFdkIsT0FBTyxJQUFJLENBQUM7U0FDYjtRQUVELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVNLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7SUFDN0IsQ0FBQztJQUVTLHFCQUFxQjtRQUM3QixPQUFPLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7O0tBS0M7SUFDUyxrQkFBa0IsQ0FBQyxXQUFvQjtRQUMvQyxPQUFPLE9BQU8sV0FBVyxLQUFLLFFBQVEsSUFBSSxXQUFXLENBQUMsSUFBSSxFQUFFLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztJQUMxRSxDQUFDO0NBQ0Y7QUEvQkQsMEJBK0JDIn0=