"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareSecuritySchemes = exports.getAllAuthorizers = exports.serializeAsAuthorizerReference = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const authorizers_1 = require("../authorizers");
const predicates_1 = require("../authorizers/predicates");
const utils_1 = require("./utils");
// Regex to match against a single header identity source
const SINGLE_HEADER_IDENTITY_SOURCE_REGEX = /^method.request.header.(?<header>[^\.\s,]+)$/;
/**
 * Build a serialized reference to an authorizer for use in an api method
 * @param authorizer the author to serialize
 */
exports.serializeAsAuthorizerReference = (authorizer) => {
    if (predicates_1.isIamAuthorizer(authorizer) ||
        predicates_1.isCustomAuthorizer(authorizer) ||
        predicates_1.isCognitoAuthorizer(authorizer)) {
        return {
            authorizerId: authorizer.authorizerId,
            authorizationScopes: authorizer.authorizationScopes,
        };
    }
    // NONE authorizer is just returned as undefined
    return undefined;
};
/**
 * Create an OpenAPI security scheme definition for an iam authorizer
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/api-gateway-swagger-extensions-authtype.html
 */
const iamSecurityScheme = () => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": "awsSigv4",
});
/**
 * Create an OpenAPI security scheme definition for a cognito authorizer
 * @param authorizer cognito authorizer
 */
const cognitoSecurityScheme = (authorizer) => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": authorizer.authorizationType,
    "x-amazon-apigateway-authorizer": {
        type: aws_apigateway_1.AuthorizationType.COGNITO,
        providerARNs: authorizer.userPools.map((pool) => pool.userPoolArn),
    },
});
/**
 * Create an OpenAPI security scheme definition for a custom authorizer
 * @param scope the scope in which the api construct is defined
 * @param authorizer custom authorizer
 */
const customSecurityScheme = (scope, authorizer) => {
    const singleHeaderMatch = authorizer.identitySource.match(SINGLE_HEADER_IDENTITY_SOURCE_REGEX);
    const singleHeader = singleHeaderMatch
        ? singleHeaderMatch.groups.header
        : undefined;
    // Token type must use a single header
    if (!singleHeader && authorizer.type === authorizers_1.CustomAuthorizerType.TOKEN) {
        throw new Error(`identitySource must be a single header for a ${authorizers_1.CustomAuthorizerType.TOKEN} authorizer, eg method.request.header.Authorization`);
    }
    return {
        type: "apiKey",
        in: "header",
        // When the identity source is not a single header, the value must be "Unused"
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-authorizer.html
        name: singleHeader ? singleHeader : "Unused",
        "x-amazon-apigateway-authtype": authorizer.authorizationType,
        "x-amazon-apigateway-authorizer": {
            type: authorizer.type,
            authorizerUri: utils_1.functionInvocationUri(scope, authorizer.function),
            authorizerResultTtlInSeconds: authorizer.authorizerResultTtlInSeconds,
            identitySource: authorizer.identitySource,
        },
    };
};
/**
 * Return a list of all unique authorizers used in the api
 */
exports.getAllAuthorizers = (integrations, defaultAuthorizer) => Object.values(Object.fromEntries([
    ...(defaultAuthorizer ? [defaultAuthorizer] : []),
    ...Object.values(integrations).flatMap(({ authorizer }) => authorizer ? [authorizer] : []),
].map((a) => [a.authorizerId, a])));
/**
 * Generate the security schemes section of an OpenAPI specification
 */
exports.prepareSecuritySchemes = (scope, integrations, defaultAuthorizer) => {
    // All the defined authorizers
    const allAuthorizers = exports.getAllAuthorizers(integrations, defaultAuthorizer);
    // Cognito, IAM and custom authorizers must be declared in security schemes
    return {
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => predicates_1.isCognitoAuthorizer(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            cognitoSecurityScheme(authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => predicates_1.isCustomAuthorizer(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            customSecurityScheme(scope, authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => predicates_1.isIamAuthorizer(authorizer))
            .map((authorizer) => [authorizer.authorizerId, iamSecurityScheme()])),
    };
};
//# sourceMappingURL=data:application/json;base64,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