"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
/**
 * A test stack with a half-prepared pipeline ready to add CloudFormation actions to
 */
class TestFixture extends cdk.Stack {
    constructor() {
        super();
        this.pipeline = new codepipeline.Pipeline(this, 'Pipeline');
        this.sourceStage = this.pipeline.addStage({ stageName: 'Source' });
        this.deployStage = this.pipeline.addStage({ stageName: 'Deploy' });
        this.repo = new codecommit.Repository(this, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        this.sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'Source',
            output: this.sourceOutput,
            repository: this.repo,
        });
        this.sourceStage.addAction(source);
    }
}
module.exports = {
    'CreateChangeSetAction can be used to make a change set from a CodePipeline'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'MagicPipeline');
        const changeSetExecRole = new aws_iam_1.Role(stack, 'ChangeSetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cloudformation.amazonaws.com'),
        });
        /** Source! */
        const repo = new codecommit.Repository(stack, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository: repo,
            trigger: cpactions.CodeCommitTrigger.POLL,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source],
        });
        /** Build! */
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        const buildOutput = new codepipeline.Artifact('OutputYo');
        const buildAction = new cpactions.CodeBuildAction({
            actionName: 'build',
            project,
            input: sourceOutput,
            outputs: [buildOutput],
        });
        pipeline.addStage({
            stageName: 'build',
            actions: [buildAction],
        });
        /** Deploy! */
        // To execute a change set - yes, you probably do need *:* 🤷‍♀️
        changeSetExecRole.addToPolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        const stackName = 'BrelandsStack';
        const changeSetName = 'MyMagicalChangeSet';
        pipeline.addStage({
            stageName: 'prod',
            actions: [
                new cpactions.CloudFormationCreateReplaceChangeSetAction({
                    actionName: 'BuildChangeSetProd',
                    stackName,
                    changeSetName,
                    deploymentRole: changeSetExecRole,
                    templatePath: new codepipeline.ArtifactPath(buildOutput, 'template.yaml'),
                    templateConfiguration: new codepipeline.ArtifactPath(buildOutput, 'templateConfig.json'),
                    adminPermissions: false,
                }),
                new cpactions.CloudFormationExecuteChangeSetAction({
                    actionName: 'ExecuteChangeSetProd',
                    stackName,
                    changeSetName,
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'MagicPipelineArtifactsBucket212FE7BF',
                },
                'Type': 'S3',
            }, 'RoleArn': {
                'Fn::GetAtt': ['MagicPipelineRoleFB2BD6DE',
                    'Arn',
                ],
            },
            'Stages': [{
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Source',
                                'Owner': 'AWS', 'Provider': 'CodeCommit', 'Version': '1',
                            },
                            'Configuration': {
                                'RepositoryName': {
                                    'Fn::GetAtt': [
                                        'MyVeryImportantRepo11BC3EBD',
                                        'Name',
                                    ],
                                },
                                'BranchName': 'master',
                                'PollForSourceChanges': true,
                            },
                            'Name': 'source',
                            'OutputArtifacts': [
                                {
                                    'Name': 'SourceArtifact',
                                },
                            ],
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'source',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Build',
                                'Owner': 'AWS',
                                'Provider': 'CodeBuild',
                                'Version': '1',
                            },
                            'Configuration': {
                                'ProjectName': {
                                    'Ref': 'MyBuildProject30DB9D6E',
                                },
                            },
                            'InputArtifacts': [
                                {
                                    'Name': 'SourceArtifact',
                                },
                            ],
                            'Name': 'build',
                            'OutputArtifacts': [
                                {
                                    'Name': 'OutputYo',
                                },
                            ],
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'build',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Deploy',
                                'Owner': 'AWS',
                                'Provider': 'CloudFormation',
                                'Version': '1',
                            },
                            'Configuration': {
                                'ActionMode': 'CHANGE_SET_REPLACE',
                                'ChangeSetName': 'MyMagicalChangeSet',
                                'RoleArn': {
                                    'Fn::GetAtt': [
                                        'ChangeSetRole0BCF99E6',
                                        'Arn',
                                    ],
                                },
                                'StackName': 'BrelandsStack',
                                'TemplatePath': 'OutputYo::template.yaml',
                                'TemplateConfiguration': 'OutputYo::templateConfig.json',
                            },
                            'InputArtifacts': [{ 'Name': 'OutputYo' }],
                            'Name': 'BuildChangeSetProd',
                            'RunOrder': 1,
                        },
                        {
                            'ActionTypeId': {
                                'Category': 'Deploy',
                                'Owner': 'AWS',
                                'Provider': 'CloudFormation',
                                'Version': '1',
                            },
                            'Configuration': {
                                'ActionMode': 'CHANGE_SET_EXECUTE',
                                'ChangeSetName': 'MyMagicalChangeSet',
                            },
                            'Name': 'ExecuteChangeSetProd',
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'prod',
                },
            ],
        }));
        test.done();
    },
    'fullPermissions leads to admin role and full IAM capabilities with pipeline bucket+key read permissions'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: true,
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        // THEN: Role is created with full permissions
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        'Action': [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        'Effect': 'Allow',
                    },
                    {
                        'Action': [
                            'kms:Decrypt',
                            'kms:DescribeKey',
                        ],
                        'Effect': 'Allow',
                    },
                    {
                        Action: '*',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                ],
            },
            Roles: [{ Ref: roleId }],
        }));
        test.done();
    },
    'outputFileName leads to creation of output artifact'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            outputFileName: 'CreateResponse.json',
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'OutputArtifacts': [{ 'Name': 'CreateUpdate_MyStack_Artifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'replaceOnFailure switches action type'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            replaceOnFailure: true,
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'ActionMode': 'REPLACE_ON_FAILURE',
                            },
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'parameterOverrides are serialized as a string'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            parameterOverrides: {
                RepoName: stack.repo.repositoryName,
            },
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'ParameterOverrides': { 'Fn::Join': ['', [
                                            '{"RepoName":"',
                                            { 'Fn::GetAtt': ['MyVeryImportantRepo11BC3EBD', 'Name'] },
                                            '"}',
                                        ]] },
                            },
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Action service role is passed to template'(test) {
        const stack = new TestFixture();
        const importedRole = aws_iam_1.Role.fromRoleArn(stack, 'ImportedRole', 'arn:aws:iam::000000000000:role/action-role');
        const freshRole = new aws_iam_1.Role(stack, 'FreshRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('magicservice'),
        });
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'ImportedRoleAction',
            role: importedRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'FreshRoleAction',
            role: freshRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Name': 'Source',
                },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Name': 'ImportedRoleAction',
                            'RoleArn': 'arn:aws:iam::000000000000:role/action-role',
                        },
                        {
                            'Name': 'FreshRoleAction',
                            'RoleArn': {
                                'Fn::GetAtt': [
                                    'FreshRole472F6E18',
                                    'Arn',
                                ],
                            },
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Single capability is passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM,
            ],
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Multiple capabilities are passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM,
                aws_cloudformation_1.CloudFormationCapabilities.AUTO_EXPAND,
            ],
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM and AUTOEXPAND capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM,CAPABILITY_AUTO_EXPAND',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'Empty capabilities is not passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NONE,
            ],
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has no capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml',
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'cross-account CFN Pipeline': {
        'correctly creates the deployment Role in the other account'(test) {
            const app = new cdk.App();
            const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                env: {
                    account: '234567890123',
                    region: 'us-west-2',
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommit',
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'RepoName'),
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Deploy',
                        actions: [
                            new cpactions.CloudFormationCreateUpdateStackAction({
                                actionName: 'CFN',
                                stackName: 'MyStack',
                                adminPermissions: true,
                                templatePath: sourceOutput.atPath('template.yaml'),
                                account: '123456789012',
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Deploy',
                        'Actions': [
                            {
                                'Name': 'CFN',
                                'RoleArn': { 'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
                                        ]],
                                },
                                'Configuration': {
                                    'RoleArn': {
                                        'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                                ':iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
                                            ]],
                                    },
                                },
                            },
                        ],
                    },
                ],
            }));
            // the pipeline's BucketPolicy should trust both CFN roles
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
                                        ]],
                                },
                            },
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
                                        ]],
                                },
                            },
                        },
                    ],
                },
            }));
            const otherStack = app.node.findChild('cross-account-support-stack-123456789012');
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                'RoleName': 'pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
            }));
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                'RoleName': 'pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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