"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function createBitBucketAndCodeBuildPipeline(stack, props) {
    const sourceOutput = new codepipeline.Artifact();
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.BitBucketSourceAction({
                        actionName: 'BitBucket',
                        owner: 'aws',
                        repo: 'aws-cdk',
                        output: sourceOutput,
                        connectionArn: 'arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh',
                        codeBuildCloneOutput: props.codeBuildCloneOutput,
                    }),
                ],
            },
            {
                stageName: 'Build',
                actions: [
                    new cpactions.CodeBuildAction({
                        actionName: 'CodeBuild',
                        project: new codebuild.PipelineProject(stack, 'MyProject'),
                        input: sourceOutput,
                        outputs: [new codepipeline.Artifact()],
                    }),
                ],
            },
        ],
    });
}
module.exports = {
    'BitBucket source Action': {
        'produces the correct configuration when added to a pipeline'(test) {
            const stack = new core_1.Stack();
            createBitBucketAndCodeBuildPipeline(stack, {
                codeBuildCloneOutput: false,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                        'Actions': [
                            {
                                'Name': 'BitBucket',
                                'ActionTypeId': {
                                    'Owner': 'AWS',
                                    'Provider': 'CodeStarSourceConnection',
                                },
                                'Configuration': {
                                    'ConnectionArn': 'arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh',
                                    'FullRepositoryId': 'aws/aws-cdk',
                                    'BranchName': 'master',
                                },
                            },
                        ],
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
    'setting codeBuildCloneOutput=true adds permission to use the connection to the following CodeBuild Project'(test) {
        const stack = new core_1.Stack();
        createBitBucketAndCodeBuildPipeline(stack, {
            codeBuildCloneOutput: true,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                    },
                    {},
                    {},
                    {},
                    {},
                    {
                        'Action': 'codestar-connections:UseConnection',
                        'Effect': 'Allow',
                        'Resource': 'arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh',
                    },
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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