"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = stage.pipeline.node.uniqueId + 'SourceEventRule' + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(id)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic291cmNlLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNvdXJjZS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMERBQTBEO0FBQzFELHVEQUF1RDtBQUd2RCxzQ0FBbUM7QUFDbkMsc0NBQWlEO0FBRWpEOzs7R0FHRztBQUNILElBQVksU0FtQlg7QUFuQkQsV0FBWSxTQUFTO0lBQ25COzs7T0FHRztJQUNILDBCQUFhLENBQUE7SUFFYjs7O09BR0c7SUFDSCwwQkFBYSxDQUFBO0lBRWI7Ozs7T0FJRztJQUNILDhCQUFpQixDQUFBO0FBQ25CLENBQUMsRUFuQlcsU0FBUyxHQUFULGlCQUFTLEtBQVQsaUJBQVMsUUFtQnBCO0FBNkNEOzs7OztHQUtHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsZUFBTTtJQUd4QyxZQUFZLEtBQTBCO1FBQ3BDLEtBQUssQ0FBQztZQUNKLEdBQUcsS0FBSztZQUNSLFFBQVEsRUFBRSxLQUFLLENBQUMsTUFBTTtZQUN0QixRQUFRLEVBQUUsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNO1lBQzVDLFFBQVEsRUFBRSxJQUFJO1lBQ2QsY0FBYyxFQUFFLDZCQUFvQixFQUFFO1lBQ3RDLE9BQU8sRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUM7U0FDeEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1NBQ2pFO1FBRUQsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDckIsQ0FBQztJQUVELDRDQUE0QztJQUM1QyxJQUFXLFNBQVM7UUFDbEIsT0FBUTtZQUNOLFNBQVMsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsV0FBVyxDQUFDO1lBQy9DLElBQUksRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDO1NBQ3RDLENBQUM7SUFDSixDQUFDO0lBRVMsS0FBSyxDQUFDLE1BQWlCLEVBQUUsS0FBMEIsRUFBRSxPQUF1QztRQUVwRyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLFNBQVMsQ0FBQyxNQUFNLEVBQUU7WUFDM0MsTUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsUUFBUSxHQUFHLGlCQUFpQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQ25GLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsRUFBRTtnQkFDM0MsOERBQThEO2dCQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsNkRBQTZELENBQUMsQ0FBQzthQUNuSTtZQUNELElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLHVCQUF1QixDQUFDLEVBQUUsRUFBRTtnQkFDNUMsTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO2dCQUNoRCxLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQzthQUM5QixDQUFDLENBQUM7U0FDSjtRQUVELDRDQUE0QztRQUM1QyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTFDLHNDQUFzQztRQUN0QyxPQUFPLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFeEMsT0FBTztZQUNMLGFBQWEsRUFBRTtnQkFDYixRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTtnQkFDdEMsV0FBVyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUztnQkFDakMsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxDQUFDLElBQUk7YUFDbEY7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBeERELHdDQXdEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEFjdGlvbiB9IGZyb20gJy4uL2FjdGlvbic7XG5pbXBvcnQgeyBzb3VyY2VBcnRpZmFjdEJvdW5kcyB9IGZyb20gJy4uL2NvbW1vbic7XG5cbi8qKlxuICogSG93IHNob3VsZCB0aGUgUzMgQWN0aW9uIGRldGVjdCBjaGFuZ2VzLlxuICogVGhpcyBpcyB0aGUgdHlwZSBvZiB0aGUge0BsaW5rIFMzU291cmNlQWN0aW9uLnRyaWdnZXJ9IHByb3BlcnR5LlxuICovXG5leHBvcnQgZW51bSBTM1RyaWdnZXIge1xuICAvKipcbiAgICogVGhlIEFjdGlvbiB3aWxsIG5ldmVyIGRldGVjdCBjaGFuZ2VzIC1cbiAgICogdGhlIFBpcGVsaW5lIGl0J3MgcGFydCBvZiB3aWxsIG9ubHkgYmVnaW4gYSBydW4gd2hlbiBleHBsaWNpdGx5IHN0YXJ0ZWQuXG4gICAqL1xuICBOT05FID0gJ05vbmUnLFxuXG4gIC8qKlxuICAgKiBDb2RlUGlwZWxpbmUgd2lsbCBwb2xsIFMzIHRvIGRldGVjdCBjaGFuZ2VzLlxuICAgKiBUaGlzIGlzIHRoZSBkZWZhdWx0IG1ldGhvZCBvZiBkZXRlY3RpbmcgY2hhbmdlcy5cbiAgICovXG4gIFBPTEwgPSAnUG9sbCcsXG5cbiAgLyoqXG4gICAqIENvZGVQaXBlbGluZSB3aWxsIHVzZSBDbG91ZFdhdGNoIEV2ZW50cyB0byBiZSBub3RpZmllZCBvZiBjaGFuZ2VzLlxuICAgKiBOb3RlIHRoYXQgdGhlIEJ1Y2tldCB0aGF0IHRoZSBBY3Rpb24gdXNlcyBuZWVkcyB0byBiZSBwYXJ0IG9mIGEgQ2xvdWRUcmFpbCBUcmFpbFxuICAgKiBmb3IgdGhlIGV2ZW50cyB0byBiZSBkZWxpdmVyZWQuXG4gICAqL1xuICBFVkVOVFMgPSAnRXZlbnRzJyxcbn1cblxuLyoqXG4gKiBUaGUgQ29kZVBpcGVsaW5lIHZhcmlhYmxlcyBlbWl0dGVkIGJ5IHRoZSBTMyBzb3VyY2UgQWN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzU291cmNlVmFyaWFibGVzIHtcbiAgLyoqIFRoZSBpZGVudGlmaWVyIG9mIHRoZSBTMyB2ZXJzaW9uIG9mIHRoZSBvYmplY3QgdGhhdCB0cmlnZ2VyZWQgdGhlIGJ1aWxkLiAqL1xuICByZWFkb25seSB2ZXJzaW9uSWQ6IHN0cmluZztcblxuICAvKiogVGhlIGUtdGFnIG9mIHRoZSBTMyB2ZXJzaW9uIG9mIHRoZSBvYmplY3QgdGhhdCB0cmlnZ2VyZWQgdGhlIGJ1aWxkLiAqL1xuICByZWFkb25seSBlVGFnOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgb2YgdGhlIHtAbGluayBTM1NvdXJjZUFjdGlvbiBTMyBzb3VyY2UgQWN0aW9ufS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTM1NvdXJjZUFjdGlvblByb3BzIGV4dGVuZHMgY29kZXBpcGVsaW5lLkNvbW1vbkF3c0FjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqXG4gICAqL1xuICByZWFkb25seSBvdXRwdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcblxuICAvKipcbiAgICogVGhlIGtleSB3aXRoaW4gdGhlIFMzIGJ1Y2tldCB0aGF0IHN0b3JlcyB0aGUgc291cmNlIGNvZGUuXG4gICAqXG4gICAqIEBleGFtcGxlICdwYXRoL3RvL2ZpbGUuemlwJ1xuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0S2V5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEhvdyBzaG91bGQgQ29kZVBpcGVsaW5lIGRldGVjdCBzb3VyY2UgY2hhbmdlcyBmb3IgdGhpcyBBY3Rpb24uXG4gICAqIE5vdGUgdGhhdCBpZiB0aGlzIGlzIFMzVHJpZ2dlci5FVkVOVFMsIHlvdSBuZWVkIHRvIG1ha2Ugc3VyZSB0byBpbmNsdWRlIHRoZSBzb3VyY2UgQnVja2V0IGluIGEgQ2xvdWRUcmFpbCBUcmFpbCxcbiAgICogYXMgb3RoZXJ3aXNlIHRoZSBDbG91ZFdhdGNoIEV2ZW50cyB3aWxsIG5vdCBiZSBlbWl0dGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTM1RyaWdnZXIuUE9MTFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9ldmVudHMvbG9nLXMzLWRhdGEtZXZlbnRzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHRyaWdnZXI/OiBTM1RyaWdnZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUzMgYnVja2V0IHRoYXQgc3RvcmVzIHRoZSBzb3VyY2UgY29kZVxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0O1xufVxuXG4vKipcbiAqIFNvdXJjZSB0aGF0IGlzIHByb3ZpZGVkIGJ5IGEgc3BlY2lmaWMgQW1hem9uIFMzIG9iamVjdC5cbiAqXG4gKiBXaWxsIHRyaWdnZXIgdGhlIHBpcGVsaW5lIGFzIHNvb24gYXMgdGhlIFMzIG9iamVjdCBjaGFuZ2VzLCBidXQgb25seSBpZiB0aGVyZSBpc1xuICogYSBDbG91ZFRyYWlsIFRyYWlsIGluIHRoZSBhY2NvdW50IHRoYXQgY2FwdHVyZXMgdGhlIFMzIGV2ZW50LlxuICovXG5leHBvcnQgY2xhc3MgUzNTb3VyY2VBY3Rpb24gZXh0ZW5kcyBBY3Rpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTM1NvdXJjZUFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBTM1NvdXJjZUFjdGlvblByb3BzKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4ucHJvcHMsXG4gICAgICByZXNvdXJjZTogcHJvcHMuYnVja2V0LFxuICAgICAgY2F0ZWdvcnk6IGNvZGVwaXBlbGluZS5BY3Rpb25DYXRlZ29yeS5TT1VSQ0UsXG4gICAgICBwcm92aWRlcjogJ1MzJyxcbiAgICAgIGFydGlmYWN0Qm91bmRzOiBzb3VyY2VBcnRpZmFjdEJvdW5kcygpLFxuICAgICAgb3V0cHV0czogW3Byb3BzLm91dHB1dF0sXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuYnVja2V0S2V5Lmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdQcm9wZXJ0eSBidWNrZXRLZXkgY2Fubm90IGJlIGFuIGVtcHR5IHN0cmluZycpO1xuICAgIH1cblxuICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgfVxuXG4gIC8qKiBUaGUgdmFyaWFibGVzIGVtaXR0ZWQgYnkgdGhpcyBhY3Rpb24uICovXG4gIHB1YmxpYyBnZXQgdmFyaWFibGVzKCk6IFMzU291cmNlVmFyaWFibGVzIHtcbiAgICByZXR1cm4gIHtcbiAgICAgIHZlcnNpb25JZDogdGhpcy52YXJpYWJsZUV4cHJlc3Npb24oJ1ZlcnNpb25JZCcpLFxuICAgICAgZVRhZzogdGhpcy52YXJpYWJsZUV4cHJlc3Npb24oJ0VUYWcnKSxcbiAgICB9O1xuICB9XG5cbiAgcHJvdGVjdGVkIGJvdW5kKF9zY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgaWYgKHRoaXMucHJvcHMudHJpZ2dlciA9PT0gUzNUcmlnZ2VyLkVWRU5UUykge1xuICAgICAgY29uc3QgaWQgPSBzdGFnZS5waXBlbGluZS5ub2RlLnVuaXF1ZUlkICsgJ1NvdXJjZUV2ZW50UnVsZScgKyB0aGlzLnByb3BzLmJ1Y2tldEtleTtcbiAgICAgIGlmICh0aGlzLnByb3BzLmJ1Y2tldC5ub2RlLnRyeUZpbmRDaGlsZChpZCkpIHtcbiAgICAgICAgLy8gdGhpcyBtZWFucyBhIGR1cGxpY2F0ZSBwYXRoIGZvciB0aGUgc2FtZSBidWNrZXQgLSBlcnJvciBvdXRcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBTMyBzb3VyY2UgYWN0aW9uIHdpdGggcGF0aCAnJHt0aGlzLnByb3BzLmJ1Y2tldEtleX0nIGlzIGFscmVhZHkgcHJlc2VudCBpbiB0aGUgcGlwZWxpbmUgZm9yIHRoaXMgc291cmNlIGJ1Y2tldGApO1xuICAgICAgfVxuICAgICAgdGhpcy5wcm9wcy5idWNrZXQub25DbG91ZFRyYWlsV3JpdGVPYmplY3QoaWQsIHtcbiAgICAgICAgdGFyZ2V0OiBuZXcgdGFyZ2V0cy5Db2RlUGlwZWxpbmUoc3RhZ2UucGlwZWxpbmUpLFxuICAgICAgICBwYXRoczogW3RoaXMucHJvcHMuYnVja2V0S2V5XSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIHdlIG5lZWQgdG8gcmVhZCBmcm9tIHRoZSBzb3VyY2UgYnVja2V0Li4uXG4gICAgdGhpcy5wcm9wcy5idWNrZXQuZ3JhbnRSZWFkKG9wdGlvbnMucm9sZSk7XG5cbiAgICAvLyAuLi5hbmQgd3JpdGUgdG8gdGhlIFBpcGVsaW5lIGJ1Y2tldFxuICAgIG9wdGlvbnMuYnVja2V0LmdyYW50V3JpdGUob3B0aW9ucy5yb2xlKTtcblxuICAgIHJldHVybiB7XG4gICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgIFMzQnVja2V0OiB0aGlzLnByb3BzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICBTM09iamVjdEtleTogdGhpcy5wcm9wcy5idWNrZXRLZXksXG4gICAgICAgIFBvbGxGb3JTb3VyY2VDaGFuZ2VzOiB0aGlzLnByb3BzLnRyaWdnZXIgJiYgdGhpcy5wcm9wcy50cmlnZ2VyID09PSBTM1RyaWdnZXIuUE9MTCxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuIl19