import pytest
from xml.etree.ElementTree import tostring
from app.domain.edi.models import LineItem, TaxCategoryCode
from app.domain.edi.models import ItemType, UnitOfMeasure


@pytest.fixture
def sample_line():
    return LineItem(
        line_number=1,
        ean="5905745621044 ",  # trailing space
        supplier_item_code="ABC-123 ",
        item_description="Towar testowy",
        item_type=ItemType.CU,
        invoice_quantity=2.0,
        unit_of_measure=UnitOfMeasure.PCE,
        invoice_unit_net_price=12.34,
        tax_rate=23.0,
        tax_category_code=TaxCategoryCode.STANDARD,
        tax_amount=5.67,
        net_amount=24.68,
    )


def test_to_xml_root_and_children(sample_line):
    el = sample_line.to_xml()
    assert el.tag == "Line-Item"
    expected = {
        "LineNumber",
        "EAN",
        "SupplierItemCode",
        "ItemDescription",
        "ItemType",
        "InvoiceQuantity",
        "UnitOfMeasure",
        "InvoiceUnitNetPrice",
        "TaxRate",
        "TaxCategoryCode",
        "TaxAmount",
        "NetAmount",
    }
    got = {c.tag for c in el}
    assert got == expected, f"Unexpected child tags: {got ^ expected}"


def test_to_xml_values_and_trimming(sample_line):
    xml = tostring(sample_line.to_xml(), encoding="unicode")
    # trimming EAN / SupplierItemCode
    assert "<EAN>5905745621044</EAN>" in xml
    assert "<SupplierItemCode>ABC-123</SupplierItemCode>" in xml
    # numeric and enums
    assert "<LineNumber>1</LineNumber>" in xml
    assert "<ItemType>CU</ItemType>" in xml
    assert "<InvoiceQuantity>2.0</InvoiceQuantity>" in xml
    assert "<UnitOfMeasure>PCE</UnitOfMeasure>" in xml
    assert "<InvoiceUnitNetPrice>12.34</InvoiceUnitNetPrice>" in xml
    assert "<TaxRate>23.0</TaxRate>" in xml
    assert "<TaxCategoryCode>S</TaxCategoryCode>" in xml  # enum .value
    assert "<TaxAmount>5.67</TaxAmount>" in xml
    assert "<NetAmount>24.68</NetAmount>" in xml


def test_to_xml_allows_empty_strings_and_none():
    li = LineItem(
        line_number=2,
        ean="",
        supplier_item_code=None,  # None -> pusty element
        item_description="",
        item_type=ItemType.RC,
        invoice_quantity=0.0,
        unit_of_measure=UnitOfMeasure.KGM,
        invoice_unit_net_price=0.0,
        tax_rate=0.0,
        tax_category_code=TaxCategoryCode.NOT_APPLICABLE,
        tax_amount=0.0,
        net_amount=0.0,
    )
    xml = tostring(li.to_xml(), encoding="unicode")
    assert "<Line-Item>" in xml
    assert "<EAN />" in xml or "<EAN></EAN>" in xml
    assert "<UnitOfMeasure>KGM</UnitOfMeasure>"
    assert (
        "<SupplierItemCode />" in xml or "<SupplierItemCode></SupplierItemCode>" in xml
    )
    assert "<TaxCategoryCode>NA</TaxCategoryCode>" in xml
