from os.path import join, dirname

import pytest

from app.domain.config import Config
from app.domain.converters import Converter
from app.domain.edi.models import Invoice
from app.domain.epp.models import Document
from app.domain.epp.models import DocumentHeader
from app.domain.epp.parser import Parser
from tests.fixtures import File
from tests.fixtures import read_test_file

with open(join(dirname(__file__), "test.epp"), "r") as f:
    TEST_STRING = f.read()


# noinspection PyTypeChecker
@pytest.fixture
def converter():
    config = Config(
        input_dir=join(dirname(__file__), "input"),
        output_dir=join(dirname(__file__), "output"),
        processed_dir=join(dirname(__file__), "processed"),
        gln="GLN",
        ilns={
            "9542827640": "SELLER_ILN",
            "6461768707": "BUYER_ILN",
            "7831892648": "BUYER_ILN",
            "6340135676": "SELLER_ILN",
            "7772963191": "SELLER_ILN",
        },
        delivery_location_ilns={
            "9542827640": "DELIVERY_LOCATION_ILN",
            "6461768707": "DELIVERY_LOCATION_ILN",
            "7831892648": "DELIVERY_LOCATION_ILN",
            "6340135676": "DELIVERY_LOCATION_ILN",
            "7772963191": "DELIVERY_LOCATION_ILN",
        },
    )
    return Converter(config)


@pytest.fixture
def data():
    document = Parser().parse(TEST_STRING)
    assert len(document.document_headers) > 0, "Expected at least one document header"
    document_header = document.document_headers[0]
    return document, document_header


@pytest.fixture
def document_many():
    document = Parser().parse(read_test_file(File.MANY, encoding="cp1250"))
    assert len(document.document_headers) > 0, "Expected at least one document header"
    document.document_headers = [
        header for header in document.document_headers if header.customer_nip
    ]
    return document


@pytest.mark.usefixtures("data", "converter")
def test_returns_instance_of_edi_invoice(
    data,
    converter: Converter,
):
    document, document_header = data[0], data[1]
    r = converter.convert(document, document_header)
    assert isinstance(r, Invoice), "Expected result to be an EDI Invoice"
    with open(join(dirname(__file__), "output.xml"), "w") as output_file:
        output_file.write(r.to_xml_string())


@pytest.mark.usefixtures("document_many", "converter")
def test_convert_all_returns_list(document_many, converter):
    r = converter.convert_all(document_many)
    assert isinstance(r, list), "Expected result to be a list of EDI Invoices"


@pytest.mark.usefixtures("document_many", "converter")
def test_convert_all_returns_list_with_expected_number_of_elements(
    document_many, converter
):
    r = converter.convert_all(document_many)
    assert isinstance(r, list), "Expected result to be a list of EDI Invoices"
    assert len(r) == 3, "Expected result to be a list of 3 EDI Invoices"
