from json import JSONDecodeError
from json import loads as json_loads
from os import environ
from os import getcwd
from os.path import join
from pathlib import Path

from loguru import logger

from app.domain.config import Config
from app.infra.parser import Args

from .constants import DEFAULT_CONFIG_FILE

def get_config_path(args: Args) -> Path:
    """
    Get the path to the configuration file.
    Returns absolute path.
    """
    config = args.config or environ.get("EPP_CONFIG_FILE", DEFAULT_CONFIG_FILE)
    config_path = Path(config)

    if not Path(config).is_absolute():
        return Path(join(getcwd(), config))

    return config_path


def get_config(args: Args) -> Config:
    """
    Read the configuration from the command line arguments.
    Configuration is not validated using the ` Config.is_Valid () ` method.

    :raises SystemExit: (1) If the configuration file is not found or is not a file.
    :raises SystemExit: (2) In case of JSON decoding errors.
    """
    path: Path = get_config_path(args)

    if not path.exists():
        logger.error(f"Configuration file at {path.absolute()} not found.")
        raise SystemExit(1)

    if not path.is_file():
        logger.error(f"Configuration file at {path.resolve()} is not a file.")
        raise SystemExit(1)

    try:
        data = json_loads(path.read_text())
    except JSONDecodeError as e:
        logger.error(f"Error reading configuration file: {e}")
        raise SystemExit(2)

    try:
        return Config.from_dict(data)
    except KeyError as e:
        logger.error(f"Invalid configuration file. Missing configuration key: {e}")
        raise SystemExit(2)
