from os import environ
from sys import stderr
from loguru import logger

from .constants import DEFAULT_LOG_LEVEL
from .constants import DEFAULT_LOG_FILE
from .constants import ALLOWED_LOG_LEVELS

logger_format = "<green>{time:YYYY-MM-DD HH:mm:ss.SSS}</green> | <level>{level: <8}</level> | <level>{message}</level>"

def _get_log_level(log_level: str | None):
    """Get the log level."""
    return log_level or environ.get("EPP_LOG_LEVEL", default=DEFAULT_LOG_LEVEL)


def _get_log_file(log_file: str | None):
    """Get the log file."""
    return log_file or environ.get("EPP_LOG_FILE", default=DEFAULT_LOG_FILE)


def _validate_log_level(log_level: str):
    """Validate the log level."""
    if log_level not in ALLOWED_LOG_LEVELS:
        raise ValueError(
            f"Invalid log level: {log_level}. Log level must be one of the following: {ALLOWED_LOG_LEVELS}."
        )


def configure_logger(
    level: str | None,
    log_file: str | None = None,
):
    """
    Configures a logger with a specified logging level and optional file output.

    This function sets up the logger by removing any existing configuration,
    adding a standard output stream logger, and optionally adding a file-based
    log output. It validates the log level against the allowed log levels
    before applying the configuration.

    :param level: The desired logging level for the logger. Can be a string
        specifying a standard log level, such as 'INFO', 'DEBUG', etc. If None,
        a default level will be used.
    :param log_file: The path to a file where logs should be saved. If None,
        no file-based logging will be configured.
    :return: None
    :raises ValueError: If the provided `level` is not within the allowed log
        levels.
    """
    log_file = _get_log_file(log_file)
    log_level = _get_log_level(level)

    _validate_log_level(log_level)  # Validate log level

    logger.remove()  # Remove standard logger
    logger.add(stderr, format=logger_format, level=level, backtrace=True, diagnose=False)
    if log_file:
        logger.add(log_file, level=level, backtrace=True, diagnose=False)
