from time import sleep

from pathlib import Path

from loguru import logger
from watchfiles import Change
from watchfiles import watch

from app.domain.config import Config
from app.infra.data_processor import DataProcessor
from app.infra.file_processor import FileProcessor
from app.infra.parser import Args


class CommandHandler:
    def can_handle(self, command) -> bool: ...
    def handle(self, command) -> int: ...

class RunCommandHandler(CommandHandler):
    """ "Run" Command Handler. """
    def __init__(self, args: Args, config: Config):
        self.config = config
        self.args = args
        self.file_processor = FileProcessor(config, args)
        self.data_processor = DataProcessor(config, args, self.file_processor)

    def can_handle(self, command) -> bool:
        return command == "run"

    def handle(self, command):
        result = self.data_processor.process()
        logger.info(f"Processed {result.total_files} files.")
        return 0

class InitCommandHandler(CommandHandler):
    """ "Init" Command Handler. """
    def __init__(self, args: Args, config: Config):
        self.config = config
        self.args = args

    def can_handle(self, command) -> bool:
        return command == "init"

    def handle(self, command):
        print("Not implemented yet.")
        return 1

class TestCommandHandler(CommandHandler):
    """ "Test" Command Handler. """
    def __init__(self, args: Args, config: Config):
        self.config = config
        self.args = args

    def can_handle(self, command) -> bool:
        return command == "test"

    def handle(self, command) -> int:
        logger.info("Testing configuration.")
        try:
            self.config.is_valid(check_paths=True, throws=True)
            logger.info("Configuration is valid.")
        except Exception as e:
            logger.error(f"Error: {e}")
            return 1
        return 0

class WatchCommandHandler(CommandHandler):
    """ "Watch" Command Handler. """
    def __init__(self, args: Args, config: Config):
        self.config = config
        self.args = args
        self.file_processor = FileProcessor(config, args)
        self.data_processor = DataProcessor(config, args, self.file_processor)

    def can_handle(self, command) -> bool:
        return command == "watch"

    def _watch(self):
        logger.info("Watching for new files...")
        _root = Path(self.config.input_dir)
        for changes in watch(_root, recursive=False):
            for change, path in changes:
                if change == Change.added:
                    if not Path(path).is_file():
                        continue
                    if not Path(path).suffix == ".epp":
                        logger.warning(f"File {path.title()} is not an EPP file. The file will be ignored.")
                        continue
                    logger.debug(f"File {path.title()} detected.")
                    self.file_processor.process_file(Path(path))

    def handle(self, command):
        result = self.data_processor.process()
        logger.info(f"Processed {result.total_files} files.")
        try:
            self._watch()
        except KeyboardInterrupt:
            logger.info("Stopping watch...")
        return 0
    
class WorkCommandHandler(CommandHandler):
    """ "Work" Command Handler. """
    _default_interval_seconds = 60

    def __init__(self, args: Args, config: Config):
        self.config = config
        self.args = args
        self.file_processor = FileProcessor(config, args)
        self.data_processor = DataProcessor(config, args, self.file_processor)

    def _get_interval_seconds(self) -> int:
        return self.args.work_interval_seconds or self._default_interval_seconds
        
    def can_handle(self, command) -> bool:
        return command == "work"
    
    def handle(self, command):
        try:
            _interval = self._get_interval_seconds()
            logger.info(f"Starting worker with interval of {_interval} seconds...")
            while True:
                r = self.data_processor.process()
                if r.total_files > 0:
                    logger.info(f"Processed {r.total_files} files.")
                else:
                    logger.debug(f"Processed {r.total_files} files.")
                sleep(_interval)
        except KeyboardInterrupt:
            logger.info("Stopping worker...")
        return 0

class AddNipCommandHandler(CommandHandler):
    def __init__(self, args: Args, config: Config):
        self.config = config
        self.args = args

    def can_handle(self, command) -> bool:
        return command == "add-nip"

    def handle(self, command) -> int:
        _ = input("ILN: ")
        _ = input("Delivery location ILN: ")
        print("Not implemented yet.")
        return 1
