from app.domain.epp.models import (
    Document,
    DocumentHeader,
    Header,
    Product,
    PriceListItem,
    LineItem,
    Contractor,
    DeliveryCompletionDate,
    FullDocumentHeader,
)
from .epp_parser import parse_epp, EPP


class ParserException(Exception):
    pass


type DocumentHeaders = list[FullDocumentHeader]


class Parser:
    """Parser class to parse EPP files content into models."""

    def _get_full_document_headers(self, epp: EPP) -> list[FullDocumentHeader]:
        output: list[FullDocumentHeader] = []
        headers = epp.get_headers()
        for header in headers:
            obj = FullDocumentHeader(
                header=DocumentHeader.from_csv(header),
                line_items=self._get_line_items(epp, header),
            )
            output.append(obj)
        return output

    @staticmethod
    def _get_line_items(epp: EPP, header: tuple | dict) -> list[LineItem]:
        """Get line items from the EPP document header."""
        content = epp.headers.get(header, [])
        return [LineItem.from_csv(row) for row in content]

    @staticmethod
    def _get_items(epp: EPP) -> list[Product]:
        return [Product.from_csv(row) for row in epp.headers.get("TOWARY", [])]

    @staticmethod
    def _get_price_list_items(epp: EPP) -> list[PriceListItem]:
        return [PriceListItem.from_csv(row) for row in epp.headers.get("CENNIK", [])]

    @staticmethod
    def _get_contractors(epp: EPP) -> list[Contractor]:
        return [Contractor.from_csv(row) for row in epp.headers.get("KONTRAHENCI", [])]

    @staticmethod
    def _get_delivery_completion_dates(epp: EPP) -> list[DeliveryCompletionDate]:
        return [
            DeliveryCompletionDate.from_csv(row)
            for row in epp.headers.get("DATYZAKONCZENIA", [])
        ]

    def parse(self, file_content: str) -> Document:
        """Parse the EPP file content and return an object."""
        epp: EPP = parse_epp(file_content)

        file_header = Header.from_csv(epp.info)

        document_headers: DocumentHeaders = self._get_full_document_headers(epp)
        price_list_items = self._get_price_list_items(epp)
        items = self._get_items(epp)
        contractors = self._get_contractors(epp)
        delivery_completion_date = self._get_delivery_completion_dates(epp)

        return Document(
            file_header=file_header,
            document_headers=document_headers,
            products=items,
            price_list_items=price_list_items,
            contractors=contractors,
            delivery_completion_date=delivery_completion_date,
        )
