from typing import List, Dict, Union, Tuple, Any
import csv
import io

Key = Union[str, Tuple[Any, ...]]


class EPP:
    """
    Data model for parsed EPP export.
    - info: list of parsed values from the [INFO] section
    - headers: mapping of section keys -> list of parsed CSV rows
    """

    def __init__(self, info: List[Any], headers: Dict[Key, List[List[Any]]]):
        self.info = info
        self.headers = headers

    def get_headers(self):
        """Generator function that returns all the headers from the EPP file."""
        for key in self.headers.keys():
            if not isinstance(key, str):
                assert isinstance(
                    key, (tuple, dict)
                ), "key must be a string or a tuple or a dict"
                yield key

    def get_headers_list(self) -> list[tuple | dict]:
        _generator = self.get_headers()
        return list(_generator)


def parse_epp(raw_text: str) -> EPP:
    """Internal function to parse EPP text into an EPP object."""

    def convert_value(value: str) -> Any:
        value = value.strip()
        if value == "":
            return None
        if (value.startswith('"') and value.endswith('"')) or (
            value.startswith("'") and value.endswith("'")
        ):
            return value[1:-1]
        try:
            if "." in value:
                return float(value)
            return int(value)
        except ValueError:
            return value

    info: List[Any] = []
    headers: Dict[Key, List[List[Any]]] = {}
    current_key: Key | None = None
    mode = None  # 'INFO', 'HEADER', 'CONTENT'

    reader = csv.reader(io.StringIO(raw_text.strip()), delimiter=",", quotechar='"')

    for row in reader:
        if not row:
            continue

        # Check for section headers
        if len(row) == 1:
            line = row[0].strip()
            if line == "[INFO]":
                mode = "INFO"
                continue
            # noinspection SpellCheckingInspection
            if line == "[NAGLOWEK]":
                mode = "HEADER"
                continue
            # noinspection SpellCheckingInspection
            if line == "[ZAWARTOSC]":
                mode = "CONTENT"
                continue

        converted_row = [convert_value(v) for v in row]

        if mode == "INFO":
            info = converted_row
            mode = None
            continue

        if mode == "HEADER":
            current_key = (
                converted_row[0] if len(converted_row) == 1 else tuple(converted_row)
            )
            headers.setdefault(current_key, [])
            mode = None
            continue

        if mode == "CONTENT" and current_key is not None:
            headers[current_key].append(converted_row)

    return EPP(info=info, headers=headers)
