class Employee:
    """
    Represents an employee record.
    From section: [PRACOWNICY]

    Attributes
    ----------
    last_name : str
        (1) Nazwisko – last name.
    first_name : str
        (2) Pierwsze imię – first name.
    middle_name : str
        (3) Drugie imię – middle name.
    mother_name : str
        (4) Imię matki – mother's name.
    father_name : str
        (5) Imię ojca – father's name.
    birth_date : str
        (6) Data urodzenia – date of birth (yyyymmdd000000).
    birth_place : str
        (7) Miejsce urodzenia – place of birth.
    pesel : str
        (8) PESEL – national ID.
    tax_id : str
        (9) NIP – tax id.
    street : str
        (10) Ulica – street.
    house_no : str
        (11) Numer domu – house number.
    flat_no : str
        (12) Numer lokalu – apartment number.
    postal_code : str
        (13) Kod pocztowy – postal code.
    city : str
        (14) Miasto – city.
    commune : str
        (15) Gmina – commune.
    county : str
        (16) Powiat – county.
    voivodeship : str
        (17) Województwo – voivodeship.
    country : str
        (18) Państwo – country.
    phone : str
        (19) Telefon – phone.
    fax : str
        (20) Faks – fax.
    telex : str
        (21) Teleks – telex.
    post_office : str
        (22) Poczta – post office.
    po_box : str
        (23) Skrytka pocztowa – PO box.
    employment_date : str
        (24) Data zatrudnienia – employment date (yyyymmdd000000).
    termination_date : str
        (25) Data zwolnienia – termination date (empty if still employed).
    employment_mode : int
        (26) Sposób zatrudnienia – 0: no employment contract, 1: employment contract.
    zus_insurance_type : int
        (27) Typ ubezpieczenia ZUS – 0: pre-1999, 1: 1st pillar, 2: 1st+2nd pillar.
    analytics : str
        (28) Analityka – analytics code (5).
    """

    def __init__(
        self,
        last_name: str,
        first_name: str,
        middle_name: str,
        mother_name: str,
        father_name: str,
        birth_date: str,
        birth_place: str,
        pesel: str,
        tax_id: str,
        street: str,
        house_no: str,
        flat_no: str,
        postal_code: str,
        city: str,
        commune: str,
        county: str,
        voivodeship: str,
        country: str,
        phone: str,
        fax: str,
        telex: str,
        post_office: str,
        po_box: str,
        employment_date: str,
        termination_date: str,
        employment_mode: int,
        zus_insurance_type: int,
        analytics: str,
    ):
        self.last_name = last_name
        self.first_name = first_name
        self.middle_name = middle_name
        self.mother_name = mother_name
        self.father_name = father_name
        self.birth_date = birth_date
        self.birth_place = birth_place
        self.pesel = pesel
        self.tax_id = tax_id
        self.street = street
        self.house_no = house_no
        self.flat_no = flat_no
        self.postal_code = postal_code
        self.city = city
        self.commune = commune
        self.county = county
        self.voivodeship = voivodeship
        self.country = country
        self.phone = phone
        self.fax = fax
        self.telex = telex
        self.post_office = post_office
        self.po_box = po_box
        self.employment_date = employment_date
        self.termination_date = termination_date
        self.employment_mode = employment_mode
        self.zus_insurance_type = zus_insurance_type
        self.analytics = analytics


# noinspection SpellCheckingInspection
class TaxOffice:
    """
    Represents a tax office.
    From section: [URZEDYSKARBOWE]

    Attributes
    ----------
    name : str
        (1) Nazwa – name.
    street : str
        (2) Ulica – street.
    house_no : str
        (3) Numer domu – house number.
    city : str
        (4) Miasto – city.
    postal_code : str
        (5) Kod pocztowy – postal code.
    post_office : str
        (6) Poczta – post office.
    analytics : str
        (7) Analityka – analytics (5).
    bank_name : str
        (8) Nazwa banku – bank name.
    bank_account : str
        (9) Numer konta w banku – bank account number.
    """

    def __init__(
        self,
        name: str,
        street: str,
        house_no: str,
        city: str,
        postal_code: str,
        post_office: str,
        analytics: str,
        bank_name: str,
        bank_account: str,
    ):
        self.name = name
        self.street = street
        self.house_no = house_no
        self.city = city
        self.postal_code = postal_code
        self.post_office = post_office
        self.analytics = analytics
        self.bank_name = bank_name
        self.bank_account = bank_account


# noinspection SpellCheckingInspection
class PaymentIdentifier:
    """
    Represents a mass payment identifier (Navireo only).
    From section: [IDENTYFIKATORYPLATNOSCI]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID (with type mnemonic).
    payment_identifier : str
        (2) Identyfikator płatności – payment identifier.
    payment_identifier_short : str
        (3) Skrócony identyfikator płatności – short identifier.
    bank_account : str
        (4) Rachunek bankowy – bank account.
    """

    def __init__(
        self,
        document_full_id: str,
        payment_identifier: str,
        payment_identifier_short: str,
        bank_account: str,
    ):
        self.document_full_id = document_full_id
        self.payment_identifier = payment_identifier
        self.payment_identifier_short = payment_identifier_short
        self.bank_account = bank_account


# noinspection SpellCheckingInspection
class DeliveryCompletionDate:
    """
    Represents delivery completion / advance receipt date.
    From section: [DATYZAKONCZENIA]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    completion_date : str
        (2) Data zakończenia dostawy / otrzymania zaliczki – date (yyyymmdd000000).
    """

    def __init__(self, document_full_id: str, completion_date: str):
        self.document_full_id = document_full_id
        self.completion_date = completion_date

    @staticmethod
    def from_csv(row: list):
        return DeliveryCompletionDate(
            document_full_id=row[0],
            completion_date=row[1],
        )


# noinspection SpellCheckingInspection
class BuyerIdentificationNumber:
    """
    Represents a buyer identification number on a document.
    From section: [NUMERYIDNABYWCOW]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    id_type : int
        (2) Typ numeru identyfikacyjnego – 1: NIP, 2: REGON, 3: PESEL.
    id_value : str
        (3) Numer identyfikacyjny nabywcy – value.
    """

    def __init__(self, document_full_id: str, id_type: int, id_value: str):
        self.document_full_id = document_full_id
        self.id_type = id_type
        self.id_value = id_value


# noinspection SpellCheckingInspection
class CorrectionReason:
    """
    Represents a correction reason.
    From section: [PRZYCZYNYKOREKT]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    line_no : int
        (2) Liczba porządkowa (numer pozycji) – line number.
    reason_text : str
        (3) Opis przyczyny korekty – reason text.
    """

    def __init__(self, document_full_id: str, line_no: int, reason_text: str):
        self.document_full_id = document_full_id
        self.line_no = line_no
        self.reason_text = reason_text


# noinspection SpellCheckingInspection
class ContractorExtra:
    """
    Represents extra contractor data.
    From section: [DODATKOWEKONTRAHENTOW]

    Attributes
    ----------
    contractor_id : str
        (1) Kod identyfikacyjny kontrahenta – contractor id.
    excise_status : int
        (2) Status akcyzowy – 0: unknown, 1: intermediary, 2: final buyer.
    excise_tax_mode : int
        (3) Podatek akcyzowy – 0: do not charge, 1: per item card, 2: exemption, 3: charge.
    cash_method_flag : bool
        (4) Czy włączyć metodę kasową na dokumentach dla kontrahenta.
    reverse_charge_eligible : bool
        (5) Czy kontrahent czynny VAT (odwrotne obciążenie możliwe).
    """

    def __init__(
        self,
        contractor_id: str,
        excise_status: int,
        excise_tax_mode: int,
        cash_method_flag: bool,
        reverse_charge_eligible: bool,
    ):
        self.contractor_id = contractor_id
        self.excise_status = excise_status
        self.excise_tax_mode = excise_tax_mode
        self.cash_method_flag = cash_method_flag
        self.reverse_charge_eligible = reverse_charge_eligible


# noinspection SpellCheckingInspection
class ProductExtra:
    """
    Represents extra item data.
    From section: [DODATKOWETOWAROW]

    Attributes
    ----------
    item_id : str
        (1) Kod identyfikacyjny towaru – item id.
    excise_applies : bool
        (2) Czy towar podlega akcyzie – excise applies.
    mark_excise_on_doc : bool
        (3) Czy zaznaczać 'Podlega Akcyzie' na dokumencie.
    excise_amount_per_unit : float
        (4) Kwota akcyzy za podstawową jednostkę miary.
    vat_margin_applies : bool
        (5) Czy towar w systemie VAT marża.
    reverse_charge_applies : bool
        (6) Czy towar objęty odwrotnym obciążeniem (czynni VAT).
    """

    def __init__(
        self,
        item_id: str,
        excise_applies: bool,
        mark_excise_on_doc: bool,
        excise_amount_per_unit: float,
        vat_margin_applies: bool,
        reverse_charge_applies: bool,
    ):
        self.item_id = item_id
        self.excise_applies = excise_applies
        self.mark_excise_on_doc = mark_excise_on_doc
        self.excise_amount_per_unit = excise_amount_per_unit
        self.vat_margin_applies = vat_margin_applies
        self.reverse_charge_applies = reverse_charge_applies


# noinspection SpellCheckingInspection
class FiscalDocumentForVat:
    """
    Represents a 'fiscal for VAT' marker for a document.
    From section: [DOKUMENTYFISKALNEVAT]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    is_fiscal_for_vat : bool
        (2) Czy dokument fiskalny dla podatnika VAT – flag.
    """

    def __init__(self, document_full_id: str, is_fiscal_for_vat: bool):
        self.document_full_id = document_full_id
        self.is_fiscal_for_vat = is_fiscal_for_vat


# noinspection SpellCheckingInspection
class AdditionalCharge:
    """
    Represents an additional charge line.
    From section: [OPLATYDODATKOWE]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    line_no : int
        (2) Liczba porządkowa (numer pozycji) – line number.
    charge_code : str
        (3) Kod identyfikacyjny opłaty dodatkowej – code.
    charge_rate : float
        (4) Stawka (kwota jednostkowa) opłaty dodatkowej – unit rate.
    quantity : float
        (5) Ilość opłat z pozycji dokumentu – quantity.
    price : float
        (6) Cena opłaty z pozycji dokumentu – price.
    value : float
        (7) Wartość opłaty z pozycji dokumentu – value.
    """

    def __init__(
        self,
        document_full_id: str,
        line_no: int,
        charge_code: str,
        charge_rate: float,
        quantity: float,
        price: float,
        value: float,
    ):
        self.document_full_id = document_full_id
        self.line_no = line_no
        self.charge_code = charge_code
        self.charge_rate = charge_rate
        self.quantity = quantity
        self.price = price
        self.value = value


# noinspection SpellCheckingInspection
class SpecialCharge:
    """
    Represents a special charge line (accounting office only).
    From section: [OPLATYSPECJALNE]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    line_no : int
        (2) Liczba porządkowa (numer pozycji) – line number.
    item_id : str
        (3) Kod identyfikacyjny towaru podlegającego opłacie specjalnej.
    special_id : int
        (4) Identyfikator opłaty specjalnej – 0: recycling, 1: waste management.
    count : float
        (5) Ilość opłat specjalnych – quantity.
    unit_rate : float
        (6) Stawka (kwota jednostkowa) opłaty specjalnej.
    value : float
        (7) Wartość opłaty specjalnej – value.
    """

    def __init__(
        self,
        document_full_id: str,
        line_no: int,
        item_id: str,
        special_id: int,
        count: float,
        unit_rate: float,
        value: float,
    ):
        self.document_full_id = document_full_id
        self.line_no = line_no
        self.item_id = item_id
        self.special_id = special_id
        self.count = count
        self.unit_rate = unit_rate
        self.value = value


# noinspection SpellCheckingInspection
class MppRequirementFlag:
    """
    Represents the split payment (MPP) requirement marker.
    From section: [ZNACZNIKMPP]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    is_mpp_marked : bool
        (2) Czy zaznaczony znacznik MPP – flag.
    """

    def __init__(self, document_full_id: str, is_mpp_marked: bool):
        self.document_full_id = document_full_id
        self.is_mpp_marked = is_mpp_marked


# noinspection SpellCheckingInspection
class CnCode:
    """
    Represents a CN code for an item.
    From section: [KODYCN]

    Attributes
    ----------
    item_id : str
        (1) Kod identyfikacyjny towaru – item id.
    cn_code : str
        (2) Kod CN – CN code.
    """

    def __init__(self, item_id: str, cn_code: str):
        self.item_id = item_id
        self.cn_code = cn_code


# noinspection SpellCheckingInspection
class GtuCodes:
    """
    Represents GTU_XX flags and related JPK markers for an item/document.
    From section: [KODYGTU] (GTU_XX)

    Attributes
    ----------
    item_id : str
        (1) Kod identyfikacyjny towaru – item id.

    # GTU flags (2–28)
    gtu_01 : bool
        (2) Czy zaznaczony znacznik GTU_01.
    gtu_02 : bool
        (3) Czy zaznaczony znacznik GTU_02.
    gtu_03 : bool
        (4) Czy zaznaczony znacznik GTU_03.
    gtu_04 : bool
        (5) Czy zaznaczony znacznik GTU_04.
    gtu_05 : bool
        (6) Czy zaznaczony znacznik GTU_05.
    gtu_06 : bool
        (7) Czy zaznaczony znacznik GTU_06.
    gtu_07 : bool
        (8) Czy zaznaczony znacznik GTU_07.
    gtu_08 : bool
        (9) Czy zaznaczony znacznik GTU_08.
    gtu_09 : bool
        (10) Czy zaznaczony znacznik GTU_09.
    gtu_10 : bool
        (11) Czy zaznaczony znacznik GTU_10.
    gtu_11 : bool
        (12) Czy zaznaczony znacznik GTU_11.
    gtu_12 : bool
        (13) Czy zaznaczony znacznik GTU_12.
    gtu_13 : bool
        (14) Czy zaznaczony znacznik GTU_13.

    doc_type : int
        (29) Typ dokumentu (0 brak; 1 RO; 2 WEW.; 3 FP; 4 MK; 5 VAT_RR).
    wsto_ee_flag : bool
        (30) Czy zaznaczony znacznik WSTO_EE (od formatu 1.11).
    ied_flag : bool
        (31) Czy zaznaczony znacznik IED (od formatu 1.11).
    """

    def __init__(
        self,
        item_id: str,
        gtu_01: bool,
        gtu_02: bool,
        gtu_03: bool,
        gtu_04: bool,
        gtu_05: bool,
        gtu_06: bool,
        gtu_07: bool,
        gtu_08: bool,
        gtu_09: bool,
        gtu_10: bool,
        gtu_11: bool,
        gtu_12: bool,
        gtu_13: bool,
        doc_type: int,
        wsto_ee_flag: bool,
        ied_flag: bool,
    ):
        self.item_id = item_id
        self.gtu_01 = gtu_01
        self.gtu_02 = gtu_02
        self.gtu_03 = gtu_03
        self.gtu_04 = gtu_04
        self.gtu_05 = gtu_05
        self.gtu_06 = gtu_06
        self.gtu_07 = gtu_07
        self.gtu_08 = gtu_08
        self.gtu_09 = gtu_09
        self.gtu_10 = gtu_10
        self.gtu_11 = gtu_11
        self.gtu_12 = gtu_12
        self.gtu_13 = gtu_13
        self.doc_type = doc_type
        self.wsto_ee_flag = wsto_ee_flag
        self.ied_flag = ied_flag


# noinspection SpellCheckingInspection
class VatMarginPurchaseData:
    """
    Represents 'VAT margin purchases' linkage (accounting office only).
    From section: [ZAKUPYVATMARZA]

    Attributes
    ----------
    fm_selling_invoice_full_id : str
        (1) Pełny numer faktury marżowej sprzedażowej (FM).
    related_doc_issue_date : str
        (2) Data wystawienia dokumentu przyjęcia/zakupu powiązanego z FM.
    related_delivery_end_date : str
        (3) Data zakończenia dostawy przyjęcia/zakupu powiązanego z FM.
    related_doc_receipt_date : str
        (4) Data otrzymania dokumentu przyjęcia/zakupu powiązanego z FM.
    related_doc_type : str
        (5) Typ dokumentu powiązanego z FM (np. FZ, PZ, PW).
    related_doc_original_number : str
        (6) Numer oryginalny dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_symbol : str
        (7) Symbol kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_name_short : str
        (8) Nazwa (lub imię i nazwisko) kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_name_full : str
        (9) Pełna nazwa kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_city : str
        (10) Miejscowość kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_postal_code : str
        (11) Kod pocztowy miejscowości kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_address : str
        (12) Ulica i numer domu/lokalu kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    related_contractor_tax_id : str
        (13) Numer NIP kontrahenta z dokumentu przyjęcia/zakupu powiązanego z FM.
    purchases_sum_value : float
        (14) Suma wartości zakupów towarów i usług marżowych występujących na FM.
    currency : str
        (15) Waluta – currency.
    currency_rate : float
        (16) Kurs waluty – exchange rate.
    """

    def __init__(
        self,
        fm_selling_invoice_full_id: str,
        related_doc_issue_date: str,
        related_delivery_end_date: str,
        related_doc_receipt_date: str,
        related_doc_type: str,
        related_doc_original_number: str,
        related_contractor_symbol: str,
        related_contractor_name_short: str,
        related_contractor_name_full: str,
        related_contractor_city: str,
        related_contractor_postal_code: str,
        related_contractor_address: str,
        related_contractor_tax_id: str,
        purchases_sum_value: float,
        currency: str,
        currency_rate: float,
    ):
        self.fm_selling_invoice_full_id = fm_selling_invoice_full_id
        self.related_doc_issue_date = related_doc_issue_date
        self.related_delivery_end_date = related_delivery_end_date
        self.related_doc_receipt_date = related_doc_receipt_date
        self.related_doc_type = related_doc_type
        self.related_doc_original_number = related_doc_original_number
        self.related_contractor_symbol = related_contractor_symbol
        self.related_contractor_name_short = related_contractor_name_short
        self.related_contractor_name_full = related_contractor_name_full
        self.related_contractor_city = related_contractor_city
        self.related_contractor_postal_code = related_contractor_postal_code
        self.related_contractor_address = related_contractor_address
        self.related_contractor_tax_id = related_contractor_tax_id
        self.purchases_sum_value = purchases_sum_value
        self.currency = currency
        self.currency_rate = currency_rate


# noinspection SpellCheckingInspection
class SugarCharge:
    """
    Represents the sugar fee summary for a document.
    From section: [OPLATACUKROWA]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    total_sugar_fee : float
        (2) Kwota łącznej opłaty cukrowej na dokumencie – amount.
    """

    def __init__(self, document_full_id: str, total_sugar_fee: float):
        self.document_full_id = document_full_id
        self.total_sugar_fee = total_sugar_fee


# noinspection SpellCheckingInspection
class WstoItemSpec:
    """
    Represents item specification for WSTO transaction docs (accounting office).
    From section: [SPECYFIKACJATOWAROWAWSTO]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    name : str
        (2) Nazwa – name.
    quantity : float
        (3) Ilość – quantity.
    uom : str
        (4) Jednostka miary – unit of measure.
    """

    def __init__(self, document_full_id: str, name: str, quantity: float, uom: str):
        self.document_full_id = document_full_id
        self.name = name
        self.quantity = quantity
        self.uom = uom


# noinspection SpellCheckingInspection
class PaymentsAndPrepayments:
    """
    Represents payment/prepayment info for documents (accounting office only).
    From section: [PLATNOSCI]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    is_prepayment : bool
        (2) Czy przedpłata – flag.
    amount : float
        (3) Kwota – amount.
    currency : str
        (4) Waluta – currency code.
    date : str
        (5) Data – date (yyyymmdd000000).
    """

    def __init__(
        self,
        document_full_id: str,
        is_prepayment: bool,
        amount: float,
        currency: str,
        date: str,
    ):
        self.document_full_id = document_full_id
        self.is_prepayment = is_prepayment
        self.amount = amount
        self.currency = currency
        self.date = date


# noinspection SpellCheckingInspection
class WstoExtraInfo:
    """
    Represents extra information for WSTO/OSS transactions (accounting office).
    From section: [INFORMACJEWSTO]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    shipping_origin_country : str
        (2) Państwo rozpoczęcia wysyłki – country (name).
    shipping_origin_eu_code : str
        (3) Kod UE państwa rozpoczęcia wysyłki – 2-letter.
    shipping_origin_iso_code : str
        (4) Kod ISO państwa rozpoczęcia wysyłki – 2-letter.
    consumer_country : str
        (5) Państwo konsumenta – country (name).
    consumer_eu_code : str
        (6) Kod UE państwa konsumenta – 2-letter.
    consumer_iso_code : str
        (7) Kod ISO państwa konsumenta – 2-letter.
    extra_info : str
        (8) Informacje dodatkowe – free text.
    """

    def __init__(
        self,
        document_full_id: str,
        shipping_origin_country: str,
        shipping_origin_eu_code: str,
        shipping_origin_iso_code: str,
        consumer_country: str,
        consumer_eu_code: str,
        consumer_iso_code: str,
        extra_info: str,
    ):
        self.document_full_id = document_full_id
        self.shipping_origin_country = shipping_origin_country
        self.shipping_origin_eu_code = shipping_origin_eu_code
        self.shipping_origin_iso_code = shipping_origin_iso_code
        self.consumer_country = consumer_country
        self.consumer_eu_code = consumer_eu_code
        self.consumer_iso_code = consumer_iso_code
        self.extra_info = extra_info


# noinspection SpellCheckingInspection
class ForeignVatLink:
    """
    Links a foreign VAT rate to a country (e.g., for WSTO/OSS).
    From section: [STAWKIVATZAGRANICZNE]

    Attributes
    ----------
    vat_rate_symbol : str
        (1) Symbol stawki podatku VAT – symbol.
    vat_rate_percent : float
        (2) Wysokość stawki podatku VAT w procentach.
    is_standard_rate : bool
        (3) Czy stawka podstawowa (Prawda) lub obniżona (Fałsz).
    country : str
        (4) Państwo – country (name).
    eu_code : str
        (5) Kod UE państwa – 2-letter.
    iso_code : str
        (6) Kod ISO państwa – 2-letter.
    """

    def __init__(
        self,
        vat_rate_symbol: str,
        vat_rate_percent: float,
        is_standard_rate: bool,
        country: str,
        eu_code: str,
        iso_code: str,
    ):
        self.vat_rate_symbol = vat_rate_symbol
        self.vat_rate_percent = vat_rate_percent
        self.is_standard_rate = is_standard_rate
        self.country = country
        self.eu_code = eu_code
        self.iso_code = iso_code


# noinspection SpellCheckingInspection
class CorrectionRecognitionDate:
    """
    Represents the 'date of recognition' mode for a correction.
    From section: [DATYUJECIAKOREKT]

    Attributes
    ----------
    document_full_id : str
        (1) Pełny numer identyfikacji dokumentu – full document ID.
    recognition_type : int
        (2) Rodzaj daty ujęcia – 1: at original invoice date, 2: at correction issue date,
            3: other date (e.g., when positions differ).
    recognition_date : str
        (3) Data ujęcia – date (yyyymmdd000000).
    """

    def __init__(
        self, document_full_id: str, recognition_type: int, recognition_date: str
    ):
        self.document_full_id = document_full_id
        self.recognition_type = recognition_type
        self.recognition_date = recognition_date
