class LineItem:
    # noinspection SpellCheckingInspection
    """
    Represents a single line item.
    Follows EPP [ZAWARTOSC] order:
    - Base fields (1..22): Table 7 (Subiekt→Subiekt positions).
    - Correction extras (23..31): Table 8 (only for correction documents).

    No type validation or conversion is performed.
    """

    def __init__(
        self,
        position_number,
        item_type,
        item_code,
        discount_is_percent,
        discount_from_price,
        discount_cumulative,
        discount_locked,
        discount_value,
        discount_percent,
        unit,
        quantity_in_unit,
        quantity_in_stock_unit,
        warehouse_price,
        net_price,
        gross_price,
        vat_rate_percent,
        net_value,
        vat_value,
        gross_value,
        cost_value,
        one_time_service_description,
        one_time_service_name,
        # --- correction extras (Table 8) ---
        corr_price_primary=None,  # 23. price net or gross (depends on document mode)
        corr_price_secondary=None,  # 24. price gross or net (depends on document mode)
        corr_vat_rate_percent=None,  # 25. VAT rate %
        corr_quantity_in_unit=None,  # 26. quantity in unit
        corr_unit=None,  # 27. unit
        corr_discount_value_or_percent=None,  # 28. discount value or percent (depends on type)
        corr_net_value=None,  # 29. net value
        corr_vat_value=None,  # 30. VAT value
        corr_gross_value=None,  # 31. gross value
    ):
        # base fields (Table 7)
        self.position_number = position_number
        self.item_type = item_type
        self.item_code = item_code
        self.discount_is_percent = discount_is_percent
        self.discount_from_price = discount_from_price
        self.discount_cumulative = discount_cumulative
        self.discount_locked = discount_locked
        self.discount_value = discount_value
        self.discount_percent = discount_percent
        self.unit = unit
        self.quantity_in_unit = quantity_in_unit
        self.quantity_in_stock_unit = quantity_in_stock_unit
        self.warehouse_price = warehouse_price
        self.net_price = net_price
        self.gross_price = gross_price
        self.vat_rate_percent = vat_rate_percent
        self.net_value = net_value
        self.vat_value = vat_value
        self.gross_value = gross_value
        self.cost_value = cost_value
        self.one_time_service_description = one_time_service_description
        self.one_time_service_name = one_time_service_name

        # correction extras (Table 8)
        self.corr_price_primary = corr_price_primary
        self.corr_price_secondary = corr_price_secondary
        self.corr_vat_rate_percent = corr_vat_rate_percent
        self.corr_quantity_in_unit = corr_quantity_in_unit
        self.corr_unit = corr_unit
        self.corr_discount_value_or_percent = corr_discount_value_or_percent
        self.corr_net_value = corr_net_value
        self.corr_vat_value = corr_vat_value
        self.corr_gross_value = corr_gross_value

    @staticmethod
    def from_csv(data: list) -> "LineItem":
        if len(data) < 22:
            raise ValueError(
                f"Expected at least 22 values to build LineItem, got {len(data)}"
            )

        # base 22 fields (Table 7)
        base = data[:22]

        # correction extras (Table 8) — present only for correction documents
        extras = [None] * 9
        if len(data) >= 31:
            extras = data[22:31]

        return LineItem(
            base[0],
            base[1],
            base[2],
            base[3],
            base[4],
            base[5],
            base[6],
            base[7],
            base[8],
            base[9],
            base[10],
            base[11],
            base[12],
            base[13],
            base[14],
            base[15],
            base[16],
            base[17],
            base[18],
            base[19],
            base[20],
            base[21],
            *extras,
        )
