from abc import abstractmethod
from abc import ABC

from .epp_line_item import LineItem
from .epp_document_header_model import FullDocumentHeader
from .epp_others import DeliveryCompletionDate
from .epp_contractor import Contractor
from .epp_price_list_item import PriceListItem
from .epp_document_header_model import DocumentHeader
from .epp_header import Header
from .epp_product import Product


class ContractorContainer(ABC):
    """Abstract class to provide contractors to the classes."""

    @abstractmethod
    def get_contractors(self) -> list[Contractor]: ...

    @abstractmethod
    def get_contractor(self, contractor_id: str) -> Contractor | None: ...


class ProductContainer(ABC):
    """Abstract class to provide products to the classes."""

    @abstractmethod
    def get_product(self, item_id: str) -> Product | None: ...

    @abstractmethod
    def get_products(self) -> list[Product]: ...


class Document(
    ContractorContainer,
    ProductContainer,
):
    """
    Represents a document in the EPP communication file.

    Attributes:
        file_header: Header instance from the [INFO] section.
    """

    def __init__(
        self,
        file_header: Header,
        document_headers: list[FullDocumentHeader],
        products: list[Product],
        price_list_items: list[PriceListItem],
        contractors: list[Contractor],
        delivery_completion_date: list[DeliveryCompletionDate],
    ):
        self.file_header = file_header
        self.document_headers = document_headers
        self.products = products
        self.price_list_items = price_list_items
        self.contractors = contractors
        self.delivery_completion_date = delivery_completion_date

    def get_contractors(self) -> list[Contractor]:
        """Returns the list of contractors."""
        return self.contractors

    def get_price(self, item_id: str) -> PriceListItem | None:
        """Returns the price for the given item."""
        for item in self.price_list_items:
            if item.item_id == item_id:
                return item
        return None

    def get_barcode(self, item_id: str) -> str | None:
        """Returns the barcode for the given item."""
        for item in self.products:
            if item.item_id == item_id:
                return item.barcode
        return None

    def get_product(self, item_id: str) -> Product | None:
        """Returns the item for the given item_id."""
        for item in self.products:
            if item.item_id == item_id:
                return item
        return None

    def get_products(self) -> list[Product]:
        return self.products

    def get_contractor(self, contractor_id: str) -> Contractor | None:
        for contractor in self.contractors:
            if contractor_id == contractor.identifier_code:
                return contractor
        return None

    # noinspection PyMethodMayBeStatic
    def get_full_number(
        self, document_header: DocumentHeader | FullDocumentHeader
    ) -> str:
        """
        Get the full number of the document in the following format: '{doc_type} {full_number}'.
        It is deprecated by the way, you should use the get_full_number method from the DocumentHeader class.

        Returns:
            FS 2369/MAG/10/2025
        """
        if isinstance(document_header, FullDocumentHeader):
            return document_header.header.get_full_number()
        return document_header.get_full_number()

    def get_delivery_completion_date(
        self, document_header: DocumentHeader
    ) -> DeliveryCompletionDate | None:
        full_number = self.get_full_number(document_header)
        for delivery_completion_date in self.delivery_completion_date:
            if delivery_completion_date.document_full_id == full_number:
                return delivery_completion_date
        return None

    def get_full_document(
        self, document_header: DocumentHeader
    ) -> FullDocumentHeader | None:
        for full_document_header in self.document_headers:
            if full_document_header.get_header() == document_header:
                return full_document_header
        return None

    def get_line_items(
        self, document_header: DocumentHeader | FullDocumentHeader
    ) -> list[LineItem] | None:
        if isinstance(document_header, FullDocumentHeader):
            return document_header.get_line_items()
        full_document_header = self.get_full_document(document_header)
        return full_document_header.get_line_items() if full_document_header else None

    def get_line_items_from_all_documents(self) -> list[LineItem]:
        """
        Returns all line items from all headers.
        Please note that this method returns a list of LineItem objects from all documents in the file.
        Usually you should use the `get_line_items` method from the FullDocumentHeader class, or the `get_line_items` method from the Document class.
        """
        return [
            item for header in self.document_headers for item in header.get_line_items()
        ]
