from enum import Enum
from xml.etree.ElementTree import Element, SubElement


class TaxCategoryCode(Enum):
    EXEMPT = "E"
    STANDARD = "S"
    NOT_APPLICABLE = "NA"
    REVERSE_CHARGE = "AE"


class ItemType(Enum):
    IN = "IN"  # Jednostka handlowa
    RC = "RC"  # Opakowanie zwrotne
    CU = "CU"  # Jednostka fakturowana


class UnitOfMeasure(Enum):
    PCE = "PCE"  # sztuka
    KGM = "KGM"  # kilogram
    MTR = "MTR"  # metr
    PKG = "PKG"  # opakowanie


class InvoicePaymentMeans(Enum):
    """
    Attributes:
        CASH: Cash, code: 10
        CHECK: Check, code: 20
        WIRE_TRANSFER: Wire transfer, code: 42
        COMPENSATION: Compensation, code: 97
    """

    CASH = 10  # gotówka
    CHECK = 20  # czek
    WIRE_TRANSFER = 42  # przelew na konto bankowe
    COMPENSATION = 97  # kompensata


class LineItem:
    """
    Represents a line item in an EDI file.

    Attributes:
        item_type:
            Identyfikator opakowania zwrotnego:
            "CU" – jednostka handlowa
            "RC" – opakowanie zwrotne
            "IN" – jednostka fakturowana
    """

    def __init__(
        self,
        line_number: int,
        ean: str,
        supplier_item_code: str | None,
        item_description: str,
        item_type: ItemType,
        invoice_quantity: float,
        unit_of_measure: UnitOfMeasure,
        invoice_unit_net_price: float,
        tax_rate: float,
        tax_category_code: TaxCategoryCode,
        tax_amount: float,
        net_amount: float,
    ):

        if not isinstance(line_number, int):
            raise TypeError("line_number must be an integer")
        if not isinstance(ean, str):
            raise TypeError("ean must be a string")
        if isinstance(supplier_item_code, int):
            supplier_item_code = str(supplier_item_code)
        if supplier_item_code is not None and not isinstance(supplier_item_code, str):
            raise TypeError("supplier_item_code must be a string or null")
        if not isinstance(item_description, str):
            raise TypeError("item_description must be a string")
        if not isinstance(item_type, ItemType):
            raise TypeError("item_type must be ItemType enum")
        if not isinstance(invoice_quantity, float):
            raise TypeError("invoice_quantity must be a float.")
        if not isinstance(invoice_unit_net_price, float):
            raise TypeError("invoice_unit_net_price must be a float")
        if not isinstance(unit_of_measure, UnitOfMeasure):
            raise TypeError("unit_of_measure must be UnitOfMeasure enum")
        if not isinstance(tax_rate, float):
            raise TypeError("tax_rate must be a float")
        if not isinstance(tax_category_code, TaxCategoryCode):
            raise TypeError("tax_category_code must be TaxCategoryCode enum")
        if not isinstance(tax_amount, float):
            raise TypeError("tax_amount must be a float")
        if not isinstance(net_amount, float):
            raise TypeError("net_amount must be a float")

        self.line_number = line_number
        self.ean = ean
        self.supplier_item_code = supplier_item_code
        self.item_description = item_description
        self.item_type = item_type
        self.invoice_quantity = invoice_quantity
        self.invoice_unit_price = invoice_unit_net_price
        self.unit_of_measure = unit_of_measure
        self.tax_rate = tax_rate
        self.tax_category_code = tax_category_code
        self.tax_amount = tax_amount
        self.net_amount = net_amount

    def to_xml(self) -> Element:
        """Return XML Element <Line-Item> with child tags in the expected schema."""
        root = Element("Line-Item")

        def _add(tag, value):
            el = SubElement(root, tag)
            if isinstance(value, Enum):
                # noinspection PyTypeChecker
                el.text = value.value
            else:
                # try to keep strings tidy (e.g., EAN may have trailing space in sources)
                el.text = str(value).strip() if value is not None else ""
            return el

        _add("LineNumber", self.line_number)
        _add("EAN", self.ean)
        _add("SupplierItemCode", self.supplier_item_code)
        _add("ItemDescription", self.item_description)
        assert isinstance(
            self.item_type, ItemType
        ), "item_type must be of type ItemType"
        _add("ItemType", self.item_type.value)
        _add("InvoiceQuantity", self.invoice_quantity)
        assert isinstance(
            self.unit_of_measure, UnitOfMeasure
        ), "unit_of_measure must be of type UnitOfMeasure"
        _add("UnitOfMeasure", self.unit_of_measure.value)
        _add("InvoiceUnitNetPrice", self.invoice_unit_price)
        _add("TaxRate", self.tax_rate)
        _add("TaxCategoryCode", self.tax_category_code)
        _add("TaxAmount", self.tax_amount)
        _add("NetAmount", self.net_amount)

        return root
