from app.domain.config import Config
from app.domain.edi.models import Invoice
from app.domain.edi.models import InvoiceHeader
from app.domain.edi.models import InvoiceParties
from app.domain.edi.models import LineItem as EPPLineItem
from app.domain.edi.models import InvoiceSummary
from app.domain.edi.models import LineItem
from app.domain.epp.models import Document
from app.domain.epp.models import FullDocumentHeader

from .converters import convert_buyer
from .converters import convert_seller
from .converters import convert_header
from .converters import convert_to_invoice_summary
from .converters import convert_to_line_item


class Converter:
    """Converter class to convert data between EPP/EDI formats."""

    def __init__(self, config: Config):
        assert isinstance(config, Config), "Config must be an instance of Config class."
        self.config = config

    def _get_invoice_parties(
        self, document: Document, document_header: FullDocumentHeader
    ) -> InvoiceParties:
        """Get invoice parties from EPP document."""
        buyer = convert_buyer(document, document_header, self.config)
        seller = convert_seller(document, self.config, document_header.get_header())

        return InvoiceParties(
            buyer=buyer,
            seller=seller,
        )

    def _get_header(
        self, document: Document, document_header: FullDocumentHeader
    ) -> InvoiceHeader:
        """Get invoice header from EPP document."""
        _doc_header = document_header.get_header()
        return convert_header(document, _doc_header, self.config)

    # noinspection PyMethodMayBeStatic
    def _get_lines(
        self, document: Document, document_header: FullDocumentHeader
    ) -> list[LineItem]:
        """Get invoice items from EPP document."""
        line_items = document_header.get_line_items()

        def func(item: EPPLineItem):
            return convert_to_line_item(item, document)

        return [func(item) for item in line_items]

    # noinspection PyMethodMayBeStatic
    def _get_summary(self, document_header: FullDocumentHeader) -> InvoiceSummary:
        """Get invoice summary from EPP document."""
        return convert_to_invoice_summary(
            document_header,
        )

    def convert(self, data: Document, document_header: FullDocumentHeader) -> Invoice:
        """
        Convert EPP document to EDI invoice.
        :raises ConvertError:
        """
        return Invoice(
            header=self._get_header(data, document_header),
            parties=self._get_invoice_parties(data, document_header),
            lines=self._get_lines(data, document_header),
            summary=self._get_summary(document_header),
        )

    def convert_all(self, data: Document) -> list[Invoice]:
        """
        Convert the EPP document to a list of EDI invoices.

        :raises ConvertError:
        """
        return [self.convert(data, header) for header in data.document_headers]
