from django_filters.rest_framework import NumberFilter
from django_filters.filters import BaseInFilter

class MultiplePKFilter(
    NumberFilter,
    BaseInFilter,
):
    """
    Defines a filter class that extends `NumberFilter` and `BaseInFilter` to provide
    filtering functionality using multiple primary keys. This filter modifies the
    field name to include a default suffix and supports filtering query sets based on
    the provided values.

    By default, field_name will be appended with '__id' to ensure that the filter works.
    You can also use `raw_field_name` to specify the exact field name to use.
    """
    default_suffix = '__id'

    def get_field_name(self, field_name: str = None, raw_field_name: str = None):
        """
        Retrieve `field_name`.
        :param raw_field_name: If provided, it is returned as is.
        :param field_name:
        :return:
        """
        if raw_field_name:
            return raw_field_name

        assert isinstance(field_name, str), '"field_name" must be a string'

        if not field_name.endswith(self.default_suffix):
            field_name += self.default_suffix

        return field_name

    def __init__(
        self,
        *args,
        **kwargs,
    ):
        field_name = self.get_field_name(
            field_name=kwargs.pop('field_name', None),
            raw_field_name=kwargs.get('raw_field_name', None),
        )

        super().__init__( *args, field_name=field_name, **kwargs)
        assert kwargs.get('lookup_expr') is None, 'lookup_expr cannot be specified'
        self.lookup_expr = 'in'

    def filter(self, qs, value):
        """
        Filters a queryset based on the provided value. If the value is not present or is None,
        the original queryset is returned. If the value is of type UUID, it is transformed into
        a list. Uses the parent's filter implementation to apply the filtering logic.

        Parameters:
            qs: Any
                The queryset to be filtered.
            value: Any
                The value used to filter the queryset. If it is None, no filtering is applied.

        Returns:
            Filtered query set.
        """
        if not value:
            return qs

        if isinstance(value, int):
            value = [value]

        return super().filter(
            qs=qs,
            value=value,
        )